/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License. For a copy,
 * see http://www.gnu.org/licenses/gpl-2.0.html.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include "config.h"
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include "global.h"
#include "alternative.h"
#include "libstr.h"

#define MAX_VALUE (INT_MAX - 10) / 10

/* Convert a string to an integer.
 */
int str2int(char *str) {
	int i = 0, value = 0;

	if (str == NULL) {
		return -1;
	} else if (*str == '\0') {
		return -1;
	} else while (*(str + i) != '\0') {
		if ((*(str + i) >= '0') && (*(str + i) <= '9')) {
			if (value >= MAX_VALUE) {
				return -1;
			}
			value *= 10;
			value += (*(str + i) - '0');
			i++;
		} else {
			return -1;
		}
	}

	return value;
}

/* Remove the leading and trailing spaces in a string.
 */
char *remove_spaces(char *str) {
	int pos;

	if (str != NULL) {
		while ((*str == ' ') || (*str == '\t')) {
			str++;
		}
		pos = strlen(str) - 1;
		while (pos >= 0) {
			switch (*(str + pos)) {
				case ' ':
				case '\t':
				case '\r':
				case '\n':
					*(str + pos) = '\0';
					pos--;
					break;
				default:
					pos = -1;
			}
		}
	}

	return str;
}

/* Remove comment from a string.
 */
char *uncomment(char *str) {
	char *hash;

	if (str == NULL) {
		return NULL;
	}

	if (*str == '#') {
		*str = '\0';
		return str;
	}

	if ((hash = strstr(str, " #")) != NULL) {
		*hash = '\0';
	} else if ((hash = strstr(str, "\t#")) != NULL) {
		*hash = '\0';
	}

	return remove_spaces(str);
}

/* Covert a string to lowercase.
 */
char *strlower(char *str) {
	char *c;

	if (str != NULL) {
		c = str;
		while (*c != '\0') {
			if ((*c >= 'A') && (*c <= 'Z')) {
				*c += 32;
			}
			c++;
		}
	}

	return str;
}

/* Split a string in 2 strings.
 */
int split_string(const char *str, char **key, char **value, char c) {
	if ((str == NULL) || (key == NULL) || (value == NULL)) {
		return -1;
	}

	*key = (char*)str;
	if ((*value = strchr(*key, c)) != NULL) {
		*(*value)++ = '\0';
		*key = remove_spaces(*key);
		*value = remove_spaces(*value);

		return 0;
	}

	return -1;
}

int split_configline(const char *str, char **key, char **value) {
	int eq = 0;

	if ((str == NULL) || (key == NULL) || (value == NULL)) {
		return -1;
	}

	*key = remove_spaces((char*)str);
	*value = *key;

	while (**value != '\0') {
		if ((**value == ' ') || (**value == '=')) {
			if (**value == '=') eq++;
			**value = '\0';
			do {
				(*value)++;
				if (**value == '=') eq++;
			} while ((**value == ' ') || (**value == '='));

			if (eq > 1) return -1;
			return 0;
		}
		(*value)++;
	}

	value = NULL;

	return -1;
}

/* Parse a key/value combination.
 */
int parse_keyvalue(char *line, t_keyvalue **kvlist, char *delimiter) {
	char *value;
	t_keyvalue *prev;

	if ((line == NULL) || (kvlist == NULL) || (delimiter == NULL)) {
		return -1;
	}

	if ((value = strstr(line, delimiter)) != NULL) {
		*value = '\0';
		value += strlen(delimiter);

		prev = *kvlist;
		if ((*kvlist = (t_keyvalue*)malloc(sizeof(t_keyvalue))) == NULL) {
			return -1;
		}
		(*kvlist)->next = prev;

		(*kvlist)->value = NULL;
		if (((*kvlist)->key = strdup(remove_spaces(line))) == NULL) {
			free(*kvlist);
			return -1;
		}
		if (((*kvlist)->value = strdup(remove_spaces(value))) == NULL) {
			free((*kvlist)->key);
			free(*kvlist);
			return -1;
		}
	} else {
		return -1;
	}

	return 0;
}
