#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "child.h"
#include "log.h"

#define TIMER_OFF -1

typedef struct type_child {
	t_session *session;

	struct type_child *next;
} t_child;

typedef struct type_banned {
	long ip;
	int  timer;
	int  bantime;
	unsigned long connect_attempts;
	
	struct type_banned *next;
} t_banned;

t_child *childlist;
t_banned *banlist;
pthread_mutex_t child_mutex;
pthread_mutex_t ban_mutex;

static void insecure_kick_ip(long ip);

/* Initialize this module.
 */
void init_childmodule(void) {
	childlist = NULL;
	banlist = NULL;
	pthread_mutex_init(&child_mutex, NULL);
	pthread_mutex_init(&ban_mutex, NULL);
}

/* Add the session record of a child to the childlist.
 */
int add_child(t_session *session) {
	t_child *new;

	if ((new = (t_child*)malloc(sizeof(t_child))) != NULL) {
		new->session = session;

		pthread_mutex_lock(&child_mutex);

		new->next = childlist;
		childlist = new;

		pthread_mutex_unlock(&child_mutex);

		return 0;
	} else {
		return -1;
	}
}

/* Remove a child from the childlist.
 */
void remove_child(t_session *session, bool clear_record) {
	t_child *deadone = NULL, *list;

	pthread_mutex_lock(&child_mutex);

	if (childlist != NULL) {
		if (childlist->session == session) {
			deadone = childlist;
			childlist = childlist->next;
		} else {
			list = childlist;
			while (list->next != NULL) {
				if (list->next->session == session) {
					deadone = list->next;
					list->next = deadone->next;
					break;
				}
				list = list->next;
			}
		}
	}

	pthread_mutex_unlock(&child_mutex);

	if (deadone != NULL) {
		if (clear_record) {
			close_socket(deadone->session);
			free(deadone->session);
		}
		free(deadone);
	}
}

/* Check whether to allow or deny a new connection.
 */
int connection_allowed(long ip, int maxperip, int maxtotal) {
	bool banned = false;
	t_banned *ban;
	int perip = 0, total = 0;
	t_child *child;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->connect_attempts++;
			banned = true;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);

	if (banned) {
		return ca_BANNED;
	}

	pthread_mutex_lock(&child_mutex);

	child = childlist;
	while (child != NULL) {
		if (child->session->ip_address == ip) {
			perip++;
		}
		total++;
		child = child->next;
	}

	pthread_mutex_unlock(&child_mutex);

	if (perip < maxperip) {
		if (total < maxtotal) {
			return total;
		} else {
			return ca_TOOMUCH_TOTAL;
		}
	} else {
		return ca_TOOMUCH_PERIP;
	}
}

/* Disconnect all connected clients.
 */
void disconnect_clients(t_config *config) {
	t_child *child;
	t_directory *dir;
	int max_wait = 30;

	pthread_mutex_lock(&child_mutex);

	child = childlist;
	while (child != NULL) {
		//close_socket(child->session);
		child->session->force_quit = true;
		child = child->next;
	}

	pthread_mutex_unlock(&child_mutex);

	while ((childlist != NULL) && (max_wait-- > 0)) {
		usleep(100000);
	}

	dir = config->Directory;
	while (dir != NULL) {
		dir->Clients = 0;
		dir = dir->next;
	}
}

/* Check the request sending speed of the clients.
 */
void check_flooding(t_config *config) {
	t_child *child;

	if (config->BanOnFlooding > 0) {
		pthread_mutex_lock(&child_mutex);

		child = childlist;
		while (child != NULL) {
			if (child->session->flooding_timer > 0) {
				child->session->flooding_timer--;
			} else {
				if (child->session->flooding_counter < config->FloodingCount) {
					child->session->flooding_timer = config->FloodingTime;
					child->session->flooding_counter = 0;
				} else if (ip_allowed(child->session->ip_address, config->BanlistMask)) {
					ban_ip(child->session->ip_address, config->BanOnFlooding, false);
					log_warning(config->SystemLogfile, "Client banned because of flooding", child->session->ip_address);
					if (config->KickOnBan) {
						insecure_kick_ip(child->session->ip_address);
					} else {
						close_socket(child->session);
					}
				}
			}
			child = child->next;
		}

		pthread_mutex_unlock(&child_mutex);
	}
}

/* Extra child functions
 */
void print_childlist(FILE *fp) {
	t_child *child;
	char *host;
	unsigned char *ip_address;

	pthread_mutex_lock(&child_mutex);

	child = childlist;
	while (child != NULL) {
		fprintf(fp, "session id  : %d\n", child->session->child_id);
		ip_address = (unsigned char*)&(child->session->ip_address);
		fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
		fprintf(fp, "Socket      : %d\n", child->session->client_socket);
		fprintf(fp, "Kept alive  : %d\n", child->session->kept_alive);
		if (child->session->remote_user != NULL) {
		fprintf(fp, "Remote user : %s\n", child->session->remote_user);
		}
		if (child->session->file_on_disk != NULL) {
			fprintf(fp, "File on disk: %s\n", child->session->file_on_disk);
			if ((host = get_headerfield("Host:", child->session->headerfields)) != NULL) {
				fprintf(fp, "Host        : %s\n", host);
			}
		}
		child = child->next;
		if (child != NULL) {
			fprintf(fp, "---\n");
		}
	}

	pthread_mutex_unlock(&child_mutex);
}

/* Disconnect a client.
 */
void kick_client(int id) {
	t_child *child;
	
	pthread_mutex_lock(&child_mutex);

	child = childlist;
	while (child != NULL) {
		if (child->session->child_id == id) {
			//close_socket(child->session);
			child->session->force_quit = true;
			break;
		}
		child = child->next;
	}

	pthread_mutex_unlock(&child_mutex);
}

/* Kick an IP address.
 */
static void insecure_kick_ip(long ip) {
	t_child *child;

	child = childlist;
	while (child != NULL) {
		if (child->session->ip_address == ip) {
			//close_socket(child->session);
			child->session->force_quit = true;
		}
		child = child->next;
	}
}
void kick_ip(long ip) {
	pthread_mutex_lock(&child_mutex);

	insecure_kick_ip(ip);

	pthread_mutex_unlock(&child_mutex);
}

/* IP ban functions
 */
int ban_ip(long ip, int timer, bool kick_on_ban) {
	int retval = 0;
	t_banned *ban;
	bool new_ip = true;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = timer;
			ban->bantime = timer;
			new_ip = false;
			break;
		}
		ban = ban->next;
	}

	if (new_ip) {
		if ((ban = (t_banned*)malloc(sizeof(t_banned))) != NULL) {
			ban->ip = ip;
			ban->timer = timer;
			ban->bantime = timer;
			ban->connect_attempts = 0;
			ban->next = banlist;

			banlist = ban;
		} else {
			retval = -1;
		}
	}

	pthread_mutex_unlock(&ban_mutex);

	if (kick_on_ban) {
		kick_ip(ip);
	}

	return retval;
}

/* Reset the timer of a banned IP address.
 */
void reban_ip(long ip) {
	t_banned *ban;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = ban->bantime;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

/* Check the timers of the banlist.
 */
void check_banlist(t_config *config) {
	t_banned *ban, *prev = NULL, *next;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		next = ban->next;
		switch (ban->timer) {
			case TIMER_OFF:
				break;
			case 0:
				if (prev == NULL) {
					banlist = next;
				} else {
					prev->next = next;
				}
				log_unban(config->SystemLogfile, ban->ip, ban->connect_attempts);
				free(ban);
				ban = prev;
				break;
			default:
				ban->timer--;
		}
		prev = ban;
		ban = next;
	}
	
	pthread_mutex_unlock(&ban_mutex);
}

/* Unban an IP address.
 */
void unban_ip(long ip) {
	t_banned *ban, *prev = NULL;

	pthread_mutex_lock(&ban_mutex);
	
	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			if (prev == NULL) {
				banlist = ban->next;
			} else {
				prev->next = ban->next;
			}
			free(ban);
			break;
		}
		prev = ban;
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

/* Print the list of banned IP addresses.
 */
void print_banlist(FILE *fp) {
	t_banned *ban;
	unsigned char *ip_address;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		ip_address = (unsigned char*)&(ban->ip);
		fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
		fprintf(fp, "seconds left: %d\n", ban->timer);
		ban = ban->next;
		if (ban != NULL) {
			fprintf(fp, "---\n");
		}
	}

	pthread_mutex_unlock(&ban_mutex);
}
