#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <errno.h>
#include <pthread.h>
#include <grp.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include "mimetype.h"
#include "libfs.h"
#include "liblist.h"
#include "serverconfig.h"
#include "session.h"
#include "libstr.h"
#include "httpauth.h"
#include "send.h"
#include "child.h"
#include "target.h"
#include "log.h"
#ifdef HAVE_COMMAND
#include "command.h"
#endif
#ifdef HAVE_SSL
#include "libssl.h"
#endif
#ifdef HAVE_PLUGIN
#include "plugin.h"
#endif

#define FILENAME_BUFFER_SIZE   256
#define REQUEST_BUFFER_CHUNK  4096

#define GENERAL_ERROR           -1

#define fe_MAX_REQUESTSIZE      -2
#define fe_TIMEOUT              -3
#define fe_CLIENT_DISCONNECTED  -4
#define fe_READ_ERROR           -5
#define fe_FORCE_QUIT           -6

#define rs_NONE                  0
#define rs_QUIT_SERVER           1
#define rs_RELOAD_CONFIGURATION  2
#define rs_DISCONNECT_CLIENTS    3

volatile int received_signal = rs_NONE;

char *pidfile             = PIDFILE_DIR"/hiawatha.pid";
char *default_configdir   = CONFIG_DIR;
char *main_configfile     = "httpd.conf";
char *throttle_configfile = "throttle.conf";
char *mimetype_configfile = "mime.types";
char *homedir_configfile  = "userwebsites.conf";

char *hs_conlen = "Content-Length: ";

#ifdef DEBUG
#include "timer.c"
#endif

/* Create all logfiles with the right ownership and accessrights
 */
void touch_logfiles(t_config *config, char *basedir) {
	t_host *host;
	uid_t userid;
	gid_t groupid;

	if (config->ServerUid == (uid_t)-1) {
		userid = 0;
		groupid = 0;
	} else {
		userid = config->ServerUid;
		groupid = config->ServerGid;
	}
	touch_logfile(basedir, config->SystemLogfile, userid, groupid, 0640);
	if (config->GarbageLogfile != NULL) {
		touch_logfile(basedir, config->GarbageLogfile, userid, groupid, 0640);
	}

	host = config->first_host;
	while (host != NULL) {
		if (config->ServerUid == (uid_t)-1) {
			userid = host->UserId;
			groupid = host->GroupId;
		} else {
			userid = config->ServerUid;
			groupid = config->ServerGid;
		}
		touch_logfile(basedir, host->LogFile, userid, groupid, 0640);

		host = host->next;
	}
}

/* Handle an HTTP error.
 */
int handle_error(t_session *session, int code) {
	int retval = 200, length;
	char errorcode[4];
	t_CGIhandler *CGI;
	
	session->handling_error = true;
	session->mimetype = NULL;
	if (session->file_on_disk != NULL) {
		free(session->file_on_disk);
	}
	length = strlen(session->host->WebsiteRoot);
	if ((session->file_on_disk = (char*)malloc(length + strlen(session->host->ErrorHandler) + 4)) == NULL) { // + 3 for .gz (gzip encoding)
		return 500;
	}
	strcpy(session->file_on_disk, session->host->WebsiteRoot);
	strcpy(session->file_on_disk + length, session->host->ErrorHandler);

	if ((session->is_CGI_script = in_charlist(file_extension(session->file_on_disk), &(session->config->CGIextension))) == false) {
		CGI = session->config->CGIhandler;
		while (CGI != NULL) {
			if (in_charlist(file_extension(session->file_on_disk), &(CGI->extension))) {
				session->is_CGI_script = true;
				session->CGI_handler = CGI->handler;
				break;
			}
			CGI = CGI->next;
		}
	}

	if (session->is_CGI_script) {
		sprintf(errorcode, "%d", code);
		setenv("HTTP_GENERATED_ERROR", errorcode, 1);
		if (session->host->ErrorCode != -1) {
			code = session->host->ErrorCode;
		}
		retval = run_script(session, code);
	} else {
		retval = send_file(session);
	}

	return retval;
}

/* Handle the request from a client.
 */
int handle_request(t_session *session) {
	int retval = 200;
	t_CGIhandler *CGI;

	if ((session->is_CGI_script = in_charlist(file_extension(session->file_on_disk), &(session->config->CGIextension))) == false) {
		CGI = session->config->CGIhandler;
		while (CGI != NULL) {
			if (in_charlist(file_extension(session->file_on_disk), &(CGI->extension))) {
				session->is_CGI_script = true;
				session->CGI_handler = CGI->handler;
				break;
			}
			CGI = CGI->next;
		}
	}

	if (strcmp(session->method, "HEAD") == 0) {
		session->HEAD_request = true;
	}
	if ((strcmp(session->method, "GET") == 0) || session->HEAD_request) {
		if (session->is_CGI_script) {
			session->body = NULL;
			retval = run_script(session, 200);
		} else {
			retval = send_file(session);
		}
		if (retval == 404) {
			if (session->host->ShowIndex && (session->URI[strlen(session->URI) - 1] == '/')) {
				retval = show_index(session);
			}
		}
	} else if (strcmp(session->method, "POST") == 0) {
		if (session->is_CGI_script) {
			retval = run_script(session, 200);
		} else {
			retval = 405;
		}
	} else if (strcmp(session->method, "OPTIONS") == 0) {
		retval = handle_options_request(session);
	} else if (strcmp(session->method, "TRACE") == 0) {
		if (session->config->EnableTRACE) {
			retval = handle_trace_request(session);
		} else {
			retval = 501;
		}
	} else if (strcmp(session->method, "PUT") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "DELETE") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "CONNECT") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "PROPFIND") == 0) {
		retval = 501;
	} else {
		retval = 400;
	}

	return retval;
}

/* Read the request from a client socket.
 */
int fetch_request(t_session *session) {
	char *new_reqbuf, *strstart, *strend;
	long bytes_read, header_length = -1, content_length = -1;
	int result = 200, timer;
	fd_set read_fds;
	struct timeval select_timeout;
	bool keep_reading = true;

	select_timeout.tv_sec = 1;
	select_timeout.tv_usec = 0;
	timer = session->config->TimeForRequest;

	do {
		/* Check if requestbuffer contains a complete request.
		 */
		if (session->request != NULL) {
			if (header_length == -1) {
				if ((strstart = strstr(session->request, "\r\n\r\n")) != NULL) {
					header_length = strstart + 4 - session->request;
					session->header_length = header_length;
				}
			}
			if (header_length != -1) {
				if (content_length == -1) {
					if ((strstart = str_search(session->request, hs_conlen)) != NULL) {
						strstart += 16;
						if ((strend = strstr(strstart, "\r\n")) != NULL) {
							*strend = '\0';
							if ((content_length = str2int(strstart)) == -1) {
								break;
							}
							session->content_length = content_length;
							*strend = '\r';
						}
					} else {
						break;
					}
				}
				if (content_length > -1) {
					if (session->bytes_in_buffer >= header_length + content_length) {
						break;
					}
				}
			}
		}

		FD_ZERO(&read_fds);
		FD_SET(session->client_socket, &read_fds);
		switch (select(session->client_socket + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				if (errno != EINTR) {
					result = fe_READ_ERROR;
					keep_reading = false;
				}
				break;
			case 0:
				if (session->force_quit) {
					result = fe_FORCE_QUIT;
					keep_reading = false;
				} else if (--timer <= 0) {
					result = fe_TIMEOUT;
					keep_reading = false;
				} else {
					select_timeout.tv_sec = 1;
					select_timeout.tv_usec = 0;
				}
				break;
			default:
				if ((session->buffer_size - session->bytes_in_buffer) < 256) {
					session->buffer_size += REQUEST_BUFFER_CHUNK;
					if ((new_reqbuf = (char*)realloc(session->request, session->buffer_size)) != NULL) {
						session->request = new_reqbuf;
					} else {
						result = fe_READ_ERROR;
						keep_reading = false;
						break;
					}
				}

				/* Read from socket.
				 */
#ifdef HAVE_SSL
				if (session->via_ssl) {
					bytes_read = sslRecv(session->ssl_data, session->request + session->bytes_in_buffer, 
									session->buffer_size - session->bytes_in_buffer - 1);
				} else
#endif
					bytes_read = recv(session->client_socket, session->request + session->bytes_in_buffer,
									session->buffer_size - session->bytes_in_buffer - 1, 0);

				switch (bytes_read) {
					case -1:
						if (errno != EINTR) {
							result = fe_READ_ERROR;
							keep_reading = false;
						}
						break;
					case 0:
						result = fe_CLIENT_DISCONNECTED;
						keep_reading = false;
						break;
					default:
						session->bytes_in_buffer += bytes_read;
						*(session->request + session->bytes_in_buffer) = '\0';

						if (session->bytes_in_buffer > session->config->MaxReqSize) {
							keep_reading = false;
							result = fe_MAX_REQUESTSIZE;
							break;
						}
				}
		}
	} while (keep_reading);

	return result;
}

/* Convert the requestbuffer to a session record.
 */
int parse_request(t_session *session, int total_bytes) {
	int retval = 200;
	char *request_end, *str_end, *conn;

	request_end = session->request + total_bytes;

	/* Scan the first request line.
	 */
	session->method = str_end = session->request;
	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
	}
	if (str_end == request_end) {
		return 400;
	}

	*str_end = '\0';
	session->URI = ++str_end;

	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
		if (*str_end == '?') {
			if (session->vars == NULL) {
				session->vars = str_end + 1;
				*str_end = '\0';
			}
		}
	}
	if (str_end == request_end) {
		return 400;
	}

	*str_end = '\0';
	str_end++;
	if (strlen(str_end) < 8) {
		return 400;
	} else if (memcmp(str_end, "HTTP/", 5) != 0) {
		return 400;
	}

	session->http_version = str_end;
	str_end += 7;
	if ((session->body = strstr(str_end + 1, "\r\n\r\n")) != NULL) {
		*(session->body + 2) = '\0';
		session->body += 4;
	}
	if (str_end + 5 != session->body) {
		session->headerfields = parse_headerfields(str_end + 3);
	}

	if ((*(str_end - 1) != '.') || (*(str_end + 1) != '\r') || (*(str_end + 2) != '\n')) {
		return 400;
	} else if (*(str_end - 2) != '1') {
		return 505;
	}
	*(str_end + 1) = '\0';

	if ((conn = get_headerfield("Connection:", session->headerfields)) != NULL) {
		conn = strlower(remove_spaces(conn));
	}
	session->keep_alive = false;
	switch (*str_end) {
		case '0':
			if ((conn != NULL) && (session->kept_alive < session->config->MaxKeepAlive)) {
				if (str_compare(conn, "keep-alive") == 0) {
					session->keep_alive = true;
				}
			}
			break;
		case '1':
			if (get_headerfield("Host:", session->headerfields) == NULL) {
				retval = 400;
			} else if (session->kept_alive < session->config->MaxKeepAlive) {
				session->keep_alive = true;
				if (conn != NULL) {
					if (strcmp(conn, "close") == 0) {
						session->keep_alive = false;
					}
				}
			}
			break;
		default:
			retval = 505;
			break;
	}
	if (session->keep_alive) {
		session->kept_alive++;
	}

	return retval;
}

/* Convert the request URI to a filename.
 */
int URI_to_path(t_session *session) {
	int retval = 200, length, alias_length;
	char *strstart, *strend, *user;
	t_keyvalue *alias;
	
	/* Requested file in userdirectory?
	 */
	if ((session->host->UserWebsites) && (*(session->URI + 1) == '~')) {
		strstart = session->URI + 2;
		if ((strend = strchr(strstart, '/')) == NULL) {
			return 301;
		} else if ((user = (char*)malloc(strend - strstart + 1)) == NULL) {
			return 500;
		}
		memcpy(user, strstart, strend - strstart);
		*(user + (strend - strstart)) = '\0';
		
		if (duplicate_host(session)) {
			retval = get_homedir(session, user);
			session->host->ErrorHandler = NULL;
		} else {
			retval = 500;
		}
	} else {
		user = NULL;
	}

	/* Convert request to complete path + filename.
	 */
	if (retval == 200) {
		if ((session->file_on_disk = (char*)malloc(FILENAME_BUFFER_SIZE + 4)) != NULL) { // + 3 for .gz (gzip encoding)
			*(session->file_on_disk + FILENAME_BUFFER_SIZE) = '\0';

			/* Search for an alias.
			 */
			alias = session->host->Alias;
			while (alias != NULL) {
				alias_length = strlen(alias->key);
				if (strncmp(session->URI, alias->key, alias_length) == 0) {
					break;
				}
				alias = alias->next;
			}
			if (alias == NULL) {
				/* No alias
				 */
				strncpy(session->file_on_disk, session->host->WebsiteRoot, FILENAME_BUFFER_SIZE);
				strstart = session->URI;
				if (user != NULL) {
					strstart += strlen(user) + 2;
					free(user);
				}
				length = strlen(session->file_on_disk);
				strncpy(session->file_on_disk + length, strstart, FILENAME_BUFFER_SIZE - length);
			} else {
				/* Use alias
				 */
				strncpy(session->file_on_disk, alias->value, FILENAME_BUFFER_SIZE);
				length = strlen(session->file_on_disk);
				strncpy(session->file_on_disk + length, session->URI + alias_length, FILENAME_BUFFER_SIZE - length);
			}

		} else {
			retval = 500;
		}
	}

	return retval;
}

#ifdef HAVE_PLUGIN
/* Should the plugin be used?
 */
bool is_plugin_uri(t_session *session) {
	if (session->config->PluginURI != NULL) {
		return session->host->PluginActive && (*(session->URI) == '/') && (strcmp(session->URI + 1, session->config->PluginURI) == 0);
	} else {
		return false;
	}
}
#endif

/* Serve the client that connected to the webserver
 */
int serve_client(t_session *session) {
	int result, length;
	char *dir_config, *dir, *search, *hostname;
	t_host *host_record;

	if ((result = fetch_request(session)) != 200) {
		return result;
	} else if ((result = parse_request(session, session->header_length + session->content_length)) != 200) {
		return result;
	}

	if (session->URI != NULL) {
		parse_special_chars(session->URI, strlen(session->URI) + 1);
	}
	if (session->vars != NULL) {
		parse_special_chars(session->vars, strlen(session->vars) + 1);
	}

	session->host = session->config->first_host;
	if ((hostname = get_headerfield("Host:", session->headerfields)) != NULL) {
#ifdef HAVE_SSL
		if ((host_record = get_hostrecord(session->config->first_host, hostname, session->binding->name, session->via_ssl)) != NULL) {
#else
		if ((host_record = get_hostrecord(session->config->first_host, hostname, session->binding->name)) != NULL) {
#endif
			session->host = host_record;
		}
	}

	if (valid_URI(session->URI) == false) {
		return 404;
	} else if (client_is_rejected_bot(session)) {
		return 403;
	}
#ifdef HAVE_PLUGIN
	if (is_plugin_uri(session)) {
		switch (ip_allowed(session->ip_address, session->host->AccessList)) {
			case deny:
				return 403;
				break;
			case allow:
				break;
			case pwd:
			case unknown:
				if (permission_granted(session) == false) {
					return 401;
				}
				break;
		}
		return handle_plugin(session);
	}
#endif
	if ((result = URI_to_path(session)) != 200) {
		return result;
	}

	/* Load configfile from directories
	 */
	if ((dir_config = strdup(session->file_on_disk)) != NULL) {
		search = dir_config;
		while (*search != '\0') {
			if (*search == '/') {
				length = search - dir_config + 1;
				if ((dir = (char*)malloc(length + 16)) != NULL) {
					memcpy(dir, dir_config, length);
					strcpy(dir + length, ".hiawatha");
					if (duplicate_host(session)) {
						if (read_user_configfile(dir, session->host) == -1) {
							result = 500;
						}
					} else {
						result = 500;
					}
					free(dir);
				} else {
					result = 500;
				}
			}

			if (result == 200) {
				search++;
			} else {
				break;
			}
		}
		free(dir_config);

		if (result != 200) {
			return result;
		}
	} else {
		return 500;
	}

	switch (is_directory(session->file_on_disk)) {
		case is_dir:
			session->URI_is_dir = true;
		case no_dir:
			break;
		case no_access:
			return 403;
		case not_found:
			if ((search = get_headerfield("Accept-Encoding:", session->headerfields)) != NULL) {
				if ((strstr(search, "gzip")) != NULL) {
					session->accept_gzip = true;
				}
			}
	}

	if ((result = copy_directory_settings(session)) != 200) {
		return result;
	} else switch (ip_allowed(session->ip_address, session->host->AccessList)) {
		case deny:
			return 403;
			break;
		case allow:
			break;
		case pwd:
		case unknown:
			if (permission_granted(session) == false) {
				return 401;
			}
			break;
	}

	if (*(session->URI + strlen(session->URI) - 1) == '/') {
		length = strlen(session->file_on_disk);
		strncpy(session->file_on_disk + length, session->host->StartFile, FILENAME_BUFFER_SIZE - length);
	} else if (session->URI_is_dir) {
		return 301;
	}

	return handle_request(session);
}
	
/* Request has been handled, handle the return code.
 */
void handle_request_result(t_session *session, int result) {
	if ((result > 0) && (result != 400)) {
		log_request(session, result);
	} else {
		session->keep_alive = false;
	}

	switch (result) {
		case fe_MAX_REQUESTSIZE:
			log_garbage(session);
			send_code(session, 400);
			if ((session->config->BanOnMaxReq > 0) && ip_allowed(session->ip_address, session->config->BanlistMask)) {
				ban_ip(session->ip_address, session->config->BanOnMaxReq, session->config->KickOnBan);
				log_warning(session->config->SystemLogfile, "Client banned because of sending a too large request", session->ip_address);
			}
			break;
		case fe_TIMEOUT:
			if (session->kept_alive == 0) {
				send_code(session, 408);
				log_warning(session->config->SystemLogfile, "Timeout while waiting for request", session->ip_address);
			}
			break;
		case fe_CLIENT_DISCONNECTED:
			if (session->kept_alive == 0) {
				log_warning(session->config->SystemLogfile, "Client disconnected", session->ip_address);
			}
			break;
		case fe_READ_ERROR:
			if (errno != ECONNRESET) {
				log_warning(session->config->SystemLogfile, "Error while reading request", session->ip_address);
#ifdef DEBUG
				log_warning(session->config->SystemLogfile, strerror(errno), session->ip_address);
#endif
			}
			break;
		case fe_FORCE_QUIT:
			log_warning(session->config->SystemLogfile, "Client kicked", session->ip_address);
			break;
		case rr_SQL_INJECTION:
			log_request(session, result);
			if ((session->config->BanOnSQLi > 0) && ip_allowed(session->ip_address, session->config->BanlistMask)) {
				ban_ip(session->ip_address, session->config->BanOnSQLi, session->config->KickOnBan);
				log_warning(session->config->SystemLogfile, "Client banned because of SQL injection", session->ip_address);
			}
			send_code(session, 400);
			break;
		case 200:
			break;
		case 304:
		case 412:
		case 413:
			send_header(session, result);
			send_buffer(session, "\r\n", 2);
			break;
		case 400:
			log_garbage(session);
			send_code(session, result);
			if ((session->config->BanOnGarbage > 0) && ip_allowed(session->ip_address, session->config->BanlistMask)) {
				ban_ip(session->ip_address, session->config->BanOnGarbage, session->config->KickOnBan);
				log_warning(session->config->SystemLogfile, "Client banned because of sending garbage", session->ip_address);
			}
			break;
		case 500:
			session->keep_alive = false;
		default:
			if ((session->host->ErrorHandler == NULL) || (result == 301)) {
				if (send_code(session, result) == -1) {
					session->keep_alive = false;
				}
			} else {
				switch (handle_error(session, result)) {
					case -1:
						session->keep_alive = false;
					case 200:
						break;
					default:
						if (send_code(session, result) == -1) {
							session->keep_alive = false;
						}
				}
			}
	}

	restore_host(session);
}

/* Handle the connection of a client.
 */
void connection_handler(t_session *session) {
	int result;

#ifdef HAVE_SSL
	if (session->via_ssl) {
		session->socket_open = sslAccept(session->client_socket, &(session->ssl_data));
	} else
#endif
		session->socket_open = true;

	if (session->socket_open) {
		do {
			result = serve_client(session);
			handle_request_result(session, result);

#ifdef HAVE_COMMAND
			tell_cc_Im_busy();
#endif
			reset_session(session);
#ifdef HAVE_COMMAND
			tell_cc_Im_done();
#endif
		} while (session->keep_alive);

		close_socket(session);
	} else {
		close(session->client_socket);
	}

#ifdef HAVE_COMMAND
	tell_cc_Im_busy();
#endif
	remove_child(session, true);
#ifdef HAVE_COMMAND
	tell_cc_Im_done();
#endif
	// Client session ends here.
}

void TERM_handler() {
	received_signal = rs_QUIT_SERVER;
}

void USR1_handler() {
	received_signal = rs_RELOAD_CONFIGURATION;
}

void USR2_handler() {
	received_signal = rs_DISCONNECT_CLIENTS;
}

/* Fill a filedescriptor set with sockets.
 */
#ifndef HAVE_COMMAND
#ifdef HAVE_SSL
int fill_read_fds(fd_set *read_fds, t_bindlist *bind_http, t_bindlist *bind_https) {
#else
int fill_read_fds(fd_set *read_fds, t_bindlist *bind_http) {
#endif
#else
#ifdef HAVE_SSL
int fill_read_fds(fd_set *read_fds, t_bindlist *bind_http, t_bindlist *bind_https, t_bindlist *bind_command) {
#else
int fill_read_fds(fd_set *read_fds, t_bindlist *bind_http, t_bindlist *bind_command) {
#endif
	t_admin *admin;
#endif
	int highest_fd = 0;

	FD_ZERO(read_fds);
	while (bind_http != NULL) {
		FD_SET(bind_http->socket, read_fds);
		if (bind_http->socket > highest_fd) {
			highest_fd = bind_http->socket;
		}
		bind_http = bind_http->next;
	}
#ifdef HAVE_SSL
	while (bind_https != NULL) {
		FD_SET(bind_https->socket, read_fds);
		if (bind_https->socket > highest_fd) {
			highest_fd = bind_https->socket;
		}
		bind_https = bind_https->next;
	}
#endif
#ifdef HAVE_COMMAND
	while (bind_command != NULL) {
		FD_SET(bind_command->socket, read_fds);
		if (bind_command->socket > highest_fd) {
			highest_fd = bind_command->socket;
		}
		bind_command = bind_command->next;
	}
	
	admin = first_admin();
	while (admin != NULL) {
		FD_SET(admin->socket, read_fds);
		if (admin->socket > highest_fd) {
			highest_fd = admin->socket;
		}
		admin = next_admin();
	}
#endif

	return highest_fd;
}

/* Create a socketlist.
 */
int bind_socket(t_bindlist *binding) {
	int one;
	struct sockaddr_in saddr;
	char *ip_address;

	while (binding != NULL) {
		if ((binding->socket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
			return -1;
		}

		bzero(&saddr, sizeof(struct sockaddr_in));
		saddr.sin_family = AF_INET;
		saddr.sin_port = htons(binding->port);
		saddr.sin_addr.s_addr = binding->ip;

		one = 1;
		if (setsockopt(binding->socket, SOL_SOCKET, SO_REUSEADDR, (void*)&one, sizeof(int)) == -1) {
			return -1;
		}
		one = 1;
		if (setsockopt(binding->socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
			return -1;
		}

		if (bind(binding->socket, (struct sockaddr*)&saddr, sizeof(struct sockaddr)) == -1) {
			ip_address = (char*)&(binding->ip);
			printf("\nError binding %hhu.%hhu.%hhu.%hhu:%d\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3], binding->port);
			return -1;
		}
		binding = binding->next;
	}

	return 0;
}

/* Accept or deny an incoming connection.
 */
#ifdef HAVE_SSL
int accept_connection(t_bindlist *binding, t_config *config, bool via_ssl) {
#else
int accept_connection(t_bindlist *binding, t_config *config) {
#endif
	unsigned int       size;
	bool               kick_client;
	t_session          *session;
	struct sockaddr_in caddr;
	pthread_attr_t     child_attr;
	pthread_t          child_thread;
	int                total_conns, one;

	if ((session = (t_session*)malloc(sizeof(t_session))) == NULL) {
		return -1;
	}
	session->config = config;
	session->binding = binding;
#ifdef HAVE_SSL
	session->via_ssl = via_ssl;
#endif
	init_session(session);

	memset((void*)&caddr, 0, sizeof(struct sockaddr_in));
	size = sizeof(struct sockaddr_in);
	if ((session->client_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
		free(session);
		if (errno != EINTR) {
			return -1;
		} else {
			return 0;
		}
	}

	kick_client = true;
	memcpy((char*)&(session->ip_address), (char*)&caddr.sin_addr.s_addr, 4);
	if ((total_conns = connection_allowed(session->ip_address, config->ConnectionsPerIP, config->ConnectionsTotal)) >= 0) {
		if (total_conns < (config->ConnectionsTotal >> 2)) {
			one = 1;
			if (setsockopt(session->client_socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
				close(session->client_socket);
				free(session);
				return -1;
			}
		}

		pthread_attr_init(&child_attr);
		pthread_attr_setdetachstate(&child_attr, PTHREAD_CREATE_DETACHED);
		if (add_child(session) == 0) {
			if (pthread_create(&child_thread, &child_attr, (void*)connection_handler, (void*)session) == 0) {
				kick_client = false;
			} else {
				remove_child(session, false);
			}
		}
	} else switch (total_conns) {
		case ca_TOOMUCH_PERIP:
			if ((config->BanOnMaxPerIP > 0) && ip_allowed(session->ip_address, session->config->BanlistMask)) {
				ban_ip(session->ip_address, config->BanOnMaxPerIP, config->KickOnBan);
			}
			log_warning(config->SystemLogfile, "Maximum number of connections for IP address reached", session->ip_address);
			break;
		case ca_TOOMUCH_TOTAL:
			log_warning(config->SystemLogfile, "Maximum number of total connections reached", session->ip_address);
			break;
		case ca_BANNED:
#ifdef DEBUG
			log_warning(config->SystemLogfile, "Client kicked because of ban", session->ip_address);
#endif
			if (config->RebanDuringBan && ip_allowed(session->ip_address, session->config->BanlistMask)) {
				reban_ip(session->ip_address);
			}
			break;
	}

	if (kick_client) {
		close(session->client_socket);
		free(session);
	}

	return 0;
}

/* Run the Hiawatha webserver.
 */
void run_server(char *configdir) {
	int                quit = 0, result, highest_fd;
#ifdef HAVE_COMMAND
	t_admin            *admin;
	t_bindlist         *old_command;
	struct sockaddr_in caddr;
	int                size, admin_socket;
#endif
	t_bindlist         *binding, *old_http, *old_https;
	FILE               *fp;
	t_config           *config, *old_config = NULL;
	t_throttle         *old_throttle = NULL;
	t_mimetype         *old_mimetype = NULL;
	t_homedir          *old_homedir = NULL;
    fd_set             read_fds;
	struct timeval     select_timeout;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);

	config = default_config();
	chdir(configdir);
	if ((result = read_main_configfile(main_configfile, config)) == 0) {
		result = check_configuration(config);
	}
	if (result != 0) {
		print_configerror(main_configfile, result);
		exit(-1);
	}
	if ((config->Mimetype = read_mimetypes(mimetype_configfile)) == NULL) {
		printf("\nWarning: can't read mimetype configuration.\n");
		exit(-1);
	}
	config->Throttle = read_throttleconfig(throttle_configfile);
	if ((config->ServerRoot == NULL) && (config->HomedirSource == hiaconf)) {
		config->Homedir = read_homedirs(homedir_configfile);
	}

	result = 0;
	/* Bind ServerPort
	 */
	if (bind_socket(config->BindHTTP) == -1) {
		return;
	}
#ifdef HAVE_SSL
	if (config->BindHTTPS != NULL) {
		if (bind_socket(config->BindHTTPS) == -1) {
			return;
		} else if (sslInit(config->ServerKey, "") == false) {
			printf("\nWarning: error while initializing SSL library!\n");
			return;
		}
	}
#endif
#ifdef HAVE_COMMAND
	if (bind_socket(config->CommandPort) == -1) {
		return;
	}
#endif

#ifdef HAVE_PLUGIN
	switch (plugin_init()) {
		case 0:
			if (config->PluginURI == NULL) {
				config->PluginURI = get_plugin_uri();
			}
			break;
		default:
			printf("\nWarning: plugin initialization failed.\n");
			exit(-1);
	}
#endif

	if ((fp = fopen(pidfile, "w")) != NULL) {
		fprintf(fp, "%d\n", getpid());
		fclose(fp);
	}
	touch_logfiles(config, config->ServerRoot);
	tzset();

	/* Security
	 */
	if (config->ServerRoot != NULL) {
		if ((chdir(config->ServerRoot) == -1) || (chroot(config->ServerRoot) == -1)) {
			printf("\nWarning: can't chroot to %s!\n", config->ServerRoot);
			exit(-1);
		}
	} else {
		chdir("/");
	}
	if ((getuid() == 0) && (config->ServerUid != (uid_t)-1)) {
		if ((setgroups(config->Groups.number, config->Groups.array) == -1) ||
			(setgid(config->ServerGid) == -1) ||
			(setuid(config->ServerUid) == -1)) {
				printf("\nWarning: can't change uid/gid.\n");
				exit(-1);
		}
	}

	log_string(config->SystemLogfile, "Hiawatha v"VERSION" started");

	signal(SIGTERM, TERM_handler);
	signal(SIGUSR1, USR1_handler);
	signal(SIGUSR2, USR2_handler);

	binding = config->BindHTTP;
	while (binding != NULL) {
		listen(binding->socket, 16);
		binding = binding->next;
	}
#ifdef HAVE_SSL
	binding = config->BindHTTPS;
	while (binding != NULL) {
		listen(binding->socket, 16);
		binding = binding->next;
	}
#endif
#ifdef HAVE_COMMAND
	binding = config->CommandPort;
	while (binding != NULL) {
		listen(binding->socket, 1);
		binding = binding->next;
	}
#endif

	select_timeout.tv_sec = 1;
	select_timeout.tv_usec = 0;

	init_childmodule();
#ifdef HAVE_COMMAND
	init_commandmodule();
#endif

	do {
#ifdef HAVE_SSL
#ifdef HAVE_COMMAND
		highest_fd = fill_read_fds(&read_fds, config->BindHTTP, config->BindHTTPS, config->CommandPort);
#else
		highest_fd = fill_read_fds(&read_fds, config->BindHTTP, config->BindHTTPS);
#endif
#else
#ifdef HAVE_COMMAND
		highest_fd = fill_read_fds(&read_fds, config->BindHTTP, config->CommandPort);
#else
		highest_fd = fill_read_fds(&read_fds, config->BindHTTP);
#endif
#endif
		switch ((result = select(highest_fd + 1, &read_fds, NULL, NULL, &select_timeout))) {
			case -1:
				if (errno != EINTR) {
#ifdef DEBUG
					log_error(config->SystemLogfile, errno);
#endif
					log_string(config->SystemLogfile, "Fatal error selecting connection");
					usleep(1000);
				}
				break;
			case 0:
				select_timeout.tv_sec = 1;
				select_timeout.tv_usec = 0;

				check_banlist(config);
#ifdef HAVE_COMMAND
				check_adminlist();
#endif
				check_flooding(config);
#ifdef HAVE_PLUGIN
				plugin_timer();
#endif
				switch (received_signal) {
					case rs_NONE:
						break;
					case rs_QUIT_SERVER:
#ifdef DEBUG
						log_string(config->SystemLogfile, "Received TERM signal");
#endif
						quit = 1;
						break;
					case rs_RELOAD_CONFIGURATION:
						disconnect_clients(config);

						old_config = config;
						old_mimetype = config->Mimetype;
						old_throttle = config->Throttle;
						old_homedir = config->Homedir;

						old_http = config->BindHTTP;
#ifdef HAVE_SSL
						old_https = config->BindHTTPS;
#endif
#ifdef HAVE_COMMAND
						old_command = config->CommandPort;
#endif

						chdir(default_configdir);

						if ((config = default_config()) != NULL) {
							if ((read_main_configfile(main_configfile, config)) == 0) {
								remove_config(old_config);
								remove_bindlist(config->BindHTTP);
#ifdef HAVE_SSL
								remove_bindlist(config->BindHTTPS);
#endif
#ifdef HAVE_COMMAND
								remove_bindlist(config->CommandPort);
#endif
							} else {
								remove_config(config);
								config = old_config;
								log_string(config->SystemLogfile, "Warning, couldn't reload the configurationfile!");
								received_signal = rs_NONE;
								break;
							}
						}

						config->BindHTTP = old_http;
#ifdef HAVE_SSL
						config->BindHTTPS = old_https;
#endif
#ifdef HAVE_COMMAND
						config->CommandPort = old_command;
#endif

						if ((config->Mimetype = read_mimetypes(mimetype_configfile)) != NULL) {
							remove_mimetypes(old_mimetype);
						} else {
							config->Mimetype = old_mimetype;
						}

						if ((config->Throttle = read_throttleconfig(throttle_configfile)) != NULL) {
							remove_throttleconfig(old_throttle);
						} else {
							config->Throttle = old_throttle;
						}

						if (config->ServerRoot == NULL) {
							if ((config->Homedir = read_homedirs(homedir_configfile)) != NULL) {
								remove_homedirs(old_homedir);
							} else {
								config->Homedir = old_homedir;
							}
						}

						touch_logfiles(config, NULL);
						chdir("/");
						log_string(config->SystemLogfile, "Configuration successfully reloaded");
						received_signal = rs_NONE;
						break;
					case rs_DISCONNECT_CLIENTS:
						disconnect_clients(config);
						received_signal = rs_NONE;
						break;
				}
				break;
			default:
#ifdef HAVE_COMMAND
				/* Connected admins
				 */
				admin = first_admin();
				while (admin != NULL) {
					if (FD_ISSET(admin->socket, &read_fds)) {
						switch (handle_admin(admin, config)) {
							case cc_DISCONNECT:
								remove_admin(admin->socket);
								break;
							case cc_SHUTDOWN:
#ifdef DEBUG
								log_string(config->SystemLogfile, "Shutdown by admin");
#endif
								quit = 1;
								break;
							case cc_RELOADCONF:
								received_signal = rs_RELOAD_CONFIGURATION;
								break;
						}
					}
					admin = next_admin();
				}
#endif

				/* Normal port
				 */
				binding = config->BindHTTP;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
#ifdef HAVE_SSL
						if (accept_connection(binding, config, false) != 0) {
#else
						if (accept_connection(binding, config) != 0) {
#endif
							log_string(config->SystemLogfile, "Fatal error accepting connection");
							usleep(1000);
							break;

						}
					}
					binding = binding->next;
				}

#ifdef HAVE_SSL
				/* SSL port
				 */
				binding = config->BindHTTPS;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						if (accept_connection(binding, config, true) != 0) {
							log_string(config->SystemLogfile, "Fatal error accepting SSL connection");
							usleep(1000);
							break;
						}
					}
					binding = binding->next;
				}
#endif

#ifdef HAVE_COMMAND
				/* Command port
				 */
				binding = config->CommandPort;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						size = sizeof(struct sockaddr_in);
						if ((admin_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
							if (errno != EINTR) {
								log_string(config->SystemLogfile, "Fatal error accepting CommandChannel connection");
								usleep(1000);
								break;
							}
						} else {
							if (add_admin(admin_socket) == -1) {
								close(admin_socket);
							}
						}
					}
					binding = binding->next;
				}
#endif
		}
	} while (quit == 0);

	disconnect_clients(config);
#ifdef HAVE_COMMAND
	disconnect_admins();
#endif

	binding = config->BindHTTP;
	while (binding != NULL) {
		close(binding->socket);
		binding = binding->next;
	}
#ifdef HAVE_SSL
	if ((binding = config->BindHTTPS) != NULL) {
		while (binding != NULL) {
			close(binding->socket);
			binding = binding->next;
		}
		sslCleanup();
	}
#endif
#ifdef HAVE_COMMAND
	binding = config->CommandPort;
	while (binding != NULL) {
		close(binding->socket);
		binding = binding->next;
	}
#endif

	signal(SIGTERM, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);
	signal(SIGUSR2, SIG_DFL);

	log_string(config->SystemLogfile, "Hiawatha v"VERSION" stopped");

#ifdef HAVE_PLUGIN
	plugin_destroy();
#endif
}

/* Main and stuff...
 */
int main(int argc, char *argv[]) {
	char *configdir = default_configdir;

	if (argc > 1) {
		configdir = argv[1];
	}

	switch (fork()) {
		case -1:
			printf("ERROR: fork1\n");
			break;
		case 0:
			if (setsid() == -1) {
				printf("ERROR: setsid\n");
			} else switch (fork()) {
				case -1:
					printf("ERROR: fork2\n");
					break;
				case 0:
					run_server(configdir);
					break;
			}
			break;
	}

	return 0;
}
