#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <stdbool.h>
#include <dirent.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "libfs.h"
#include "libstr.h"

char *months[12] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

/* Return the size of a file.
 */
long filesize(char *filename) {
	struct stat status;

	if (stat(filename, &status) == 0) {
		return status.st_size;
	} else {
		return -1;
	}
}

/* Check the accessrights of a file. (internal use)
 */
static enum t_extbool rights_oke(struct stat *status, enum t_filetype filetype, mode_t read, mode_t exec) {
	switch (filetype) {
		case file:
			if ((status->st_mode & read) == read) {
				return yes;
			}
			break;
		case path_to:
			if ((status->st_mode & exec) == exec) {
				return yes;
			}
			break;
		case executable:
		case directory:
			if (((status->st_mode & read) == read) && ((status->st_mode & exec) == exec)) {
				return yes;
			}
			break;
	}

	return no;
}

/* Check the accessrights of a file.
 */
enum t_extbool fs_access_allowed(char *filename, uid_t uid, gid_t gid, t_groups *groups, enum t_filetype filetype) {
	enum t_extbool allowed = no;
	bool in_group = false;
	struct stat status;
	char *slash;
	int num = 0;
	gid_t *group;

	if (stat(filename, &status) == 0) {
		if (status.st_uid == uid) {
			/* Check user
			 */
			allowed = rights_oke(&status, filetype, S_IRUSR, S_IXUSR);
		} else {
			/* Check group
			 */
			if (status.st_gid == gid) {
				allowed = rights_oke(&status, filetype, S_IRGRP, S_IXGRP);
				in_group = true;
			} else if (groups != NULL) {
				group = groups->array;
				while (num < groups->number) {
					if (status.st_gid == *group) {
						allowed = rights_oke(&status, filetype, S_IRGRP, S_IXGRP);
						in_group = true;
						break;
					}
					group++;
					num++;
				}
			}

			/* Check others
			 */
			if (in_group == false) {
				allowed = rights_oke(&status, filetype, S_IROTH, S_IXOTH);
			}
		}

		if (allowed == yes) {
			slash = filename + strlen(filename);
			while ((slash != filename) && (*slash != '/')) {
				slash--;
			}
			if (slash != filename) {
				*slash = '\0';
				allowed = fs_access_allowed(filename, uid, gid, groups, path_to);
				*slash = '/';
			}
		}
	} else {
		allowed = error;
	}

	return allowed;
}

enum t_extbool contains_symlink(char *filename) {
	enum t_extbool contains = no;
	struct stat status;
	char *slash;

	if (lstat(filename, &status) == -1) {
		return error;
	} else if (((status.st_mode & S_IFMT) == S_IFLNK) && (status.st_uid != 0)) {
		return yes;
	}

	slash = filename + strlen(filename);
	while ((slash != filename) && (*slash != '/')) {
		slash--;
	}
	if (slash != filename) {
		*slash = '\0';
		contains = contains_symlink(filename);
		*slash = '/';
	}

	return contains;
}

/* Check whether a file is directory or not.
 */
enum t_isdir is_directory(char *file) {
	DIR *dp;
		
	if ((dp = opendir(file)) != NULL) {
		closedir(dp);

		return is_dir;
	} else switch (errno) {
		case EACCES:
			return no_access;
		case ENOENT:
			return not_found;
		default:
			return no_dir;
	}
}

/* Create a file with the right ownership and accessrights.
 */
void touch_logfile(char *dir, char *file, uid_t userid, gid_t groupid, mode_t mode) {
	int fd; 
	char *logfile;

	if (dir != NULL) {
		if ((logfile = (char*)malloc(strlen(dir) + strlen(file) + 1)) != NULL) {
			strcpy(logfile, dir);
			strcat(logfile, file);
		}   
	} else {
		logfile = file;
	}   
	if (logfile != NULL) {
		if ((fd = open(logfile, O_RDONLY)) == -1) {
			if ((fd = open(logfile, O_CREAT|O_APPEND)) != -1) {
				fchown(fd, userid, groupid);
				fchmod(fd, mode);
				close(fd);
			}   
		} else {
			close(fd);
		}   
	}   
	if (dir != NULL) {
		free(logfile);
	}   
}

/* Monthnumber to monthname.
 */
static short month2int(char *month) {
	int i;

	for (i = 0; i < 12; i++) {
		if (memcmp(month, months[i], 3) == 0) {
			return i;
		}
	}

	return -1;
}

/* Parse a RFC 822 datestring.
 *
 * 0    5  8   12   17       26
 * Day, dd Mon yyyy hh:mm:ss GMT
 */
static int parse_datestr(char *org_datestr, struct tm *date) {
	int result = -1;
	char *datestr;

	if (strlen(org_datestr) == 29) {
		if ((datestr = strdup(org_datestr)) != NULL) {
			do {
				if (memcmp(datestr + 3, ", ", 2) != 0) {
					break;
				} else if ((*(datestr + 7) != ' ') || (*(datestr + 11) != ' ') || (*(datestr + 16) != ' ')) {
					break;
				} else if ((*(datestr + 19) != ':') || (*(datestr + 22) != ':')) {
					break;
				} else if (memcmp(datestr + 25, " GMT", 4) != 0) {
					break;
				}

				*(datestr + 7) = *(datestr + 11) = *(datestr + 16) = *(datestr + 19) = *(datestr + 22) = *(datestr + 25) = '\0';

				if ((date->tm_mday = str2int(datestr + 5)) <= 0) {
					break;
				} else if ((date->tm_mon = month2int(datestr + 8)) == -1) {
					break;
				} else if ((date->tm_year = str2int(datestr + 12)) < 1900) {
					break;
				} else if ((date->tm_hour = str2int(datestr + 17)) == -1) {
					break;
				} else if ((date->tm_min = str2int(datestr + 20)) == -1) {
					break;
				} else if ((date->tm_sec = str2int(datestr + 23)) == -1) {
					break;
				}

				if (date->tm_mday > 31) {
					break;
				} else if (date->tm_hour > 23) {
					break;
				} else if (date->tm_min > 59) {
					break;
				} else if (date->tm_sec > 59) {
					break;
				}

				date->tm_year -= 1900;

				result = 0;
			} while (false);
			free(datestr);
		}
	}

	return result;
}

/* Check wheter a file has been modified since a certain date or not.
 */
int if_modified_since(int handle, char *datestr) {
	struct stat status;
	struct tm *fdate, rdate;
	time_t file_date, req_date;

	do {
		if (fstat(handle, &status) == -1) {
			break;
		} else if ((fdate = gmtime(&(status.st_mtime))) == NULL) {
			break;
		} else if ((file_date = mktime(fdate)) == -1) {
			break;
		} else if (parse_datestr(datestr, &rdate) == -1) {
			break;
		} else if ((req_date = mktime(&rdate)) == -1) {
			break;
		} else if (file_date > req_date) {
			return 1;
		} else {
			return 0;
		}
	} while (false);

	return -1;
}

/* Open a file (search in directory where file 'neighbour' is located if not found).
 */
FILE *fopen_neighbour(char *filename, char *mode, char *neighbour) {
	FILE *fp;
	char *file, *slash;
	int len;

	if (((fp = fopen(filename, mode)) == NULL) && (neighbour != NULL)) {
		if ((slash = strrchr(neighbour, '/')) != NULL) {
			len = slash - neighbour + 1;
			if ((file = (char*)malloc(len + strlen(filename) + 1)) != NULL) {
				memcpy(file, neighbour, len);
				strcpy(file + len, filename);
				fp = fopen(file, mode);
				free(file);
			}
		}
	}

	return fp;
}
