#include "config.h"
#ifdef HAVE_SSL
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <assert.h>
#include <pthread.h>
#include "libssl.h"

static pthread_mutex_t *locks;
static int lockCount;
/* Global SSL structures */
static BIO *stderrBio;
static SSL_CTX *sslContext;

/* SSL multithread locking callback */
static void locking_callback(int mode, int n, const char *file, int line) {
	assert(n >= 0);
	assert(n < lockCount);
	if (mode & CRYPTO_LOCK) {
		pthread_mutex_lock(&locks[n]);
	} else {
		pthread_mutex_unlock(&locks[n]);
	}
}

/* SSL thread ID callback */
static unsigned long id_callback() {
	return (unsigned long)pthread_self();
}

bool sslInit(const char* sslKeyFile, const char* sslPassword) {
	SSL_METHOD *meth;
	int i;

	/* Load the SSL algorithms and error strings (global)..*/
	SSL_library_init();
	SSL_load_error_strings(); 

	/* Wrap stderr in a BIO (an openssl IO abstraction object)..*/
	stderrBio = BIO_new_fp(stderr, BIO_NOCLOSE);
  
	/* Create the SSL context (the context is used to manage keyfiles, 
	 * CA stuff etc. so they have to be loaded only once). */
	//meth = TLSv1_method();
	meth = SSLv23_method();
	sslContext = SSL_CTX_new(meth);

	/* Don't allow SSLv2 */
	SSL_CTX_set_options(sslContext, SSL_OP_NO_SSLv2);

	/* Load certificate from keyfile..*/
	if (!SSL_CTX_use_certificate_chain_file(sslContext, sslKeyFile)) {
		fprintf(stderr, "Failed to read SSL Certificate file.\n");
		ERR_print_errors(stderrBio);
		return false;
	}

	/* Load key from keyfile..*/
	SSL_CTX_set_default_passwd_cb(sslContext, sslPasswordCB);
	SSL_CTX_set_default_passwd_cb_userdata(sslContext, (void*)sslPassword);
	if (!SSL_CTX_use_PrivateKey_file(sslContext, sslKeyFile, SSL_FILETYPE_PEM)) {
		fprintf(stderr, "Failed to read SSL key file.\n");
		ERR_print_errors(stderrBio);
		return false;
	}

	/* Initialize SSL library's thread locks. */
	lockCount = CRYPTO_num_locks();
	if (lockCount) {
		locks = malloc(lockCount * sizeof(pthread_mutex_t));
		for (i = 0; i < lockCount; i++) {
			pthread_mutex_init(&locks[i], NULL);
		}
    }
  
	/* Register locking function. */
	CRYPTO_set_locking_callback(locking_callback); 
  
	/* Register thread id callback. */
	CRYPTO_set_id_callback(id_callback);
 
	return true;
}

void sslCleanup(void) {
	int i;

	/* Free library resources */
	SSL_CTX_free(sslContext);
	BIO_free(stderrBio);
  
	/* Destroy locks. */
	for (i = 0; i < lockCount; i++) {
		pthread_mutex_destroy(&locks[i]);
	}
}

int sslPasswordCB(char *passwordBuffer, int size, int rwflag, void *data) {
	char *keyfilePassword;
	int pwLength;

	keyfilePassword = (char*)data;
	pwLength = strlen(keyfilePassword);

	if (size < pwLength) {
  		return 0;
	}
	strncpy(passwordBuffer, keyfilePassword, pwLength);

	return pwLength;
} 

bool sslAccept(int socketFD, SSL **ssl) {
	BIO *socketBio;

	/* Wrap the socket in a BIO */
	socketBio = BIO_new_socket(socketFD, 0);

	/* Create a new SSL object for this connection..*/
	*ssl = SSL_new(sslContext);

	/* Attach the bio to the ssl object.. */
	SSL_set_bio(*ssl, socketBio, socketBio);

	/* The moment of truth.. (with crappy error handling!) */
	switch (SSL_get_error(*ssl, SSL_accept(*ssl))) {
		case SSL_ERROR_NONE:
			return true;
#ifdef DEBUG
		case SSL_ERROR_ZERO_RETURN:
			fprintf(stderr, "SSL zero return error\n");
			break;
			
		case SSL_ERROR_WANT_READ:
			fprintf(stderr, "SSL_ERROR_WANT_READ\n");
			break;
			
		case SSL_ERROR_WANT_WRITE:
			fprintf(stderr, "SSL_ERROR_WANT_WRITE\n");
			break;
			
		case SSL_ERROR_WANT_CONNECT:
			fprintf(stderr, "SSL_ERROR_WANT_CONNECT\n");
			break;
			
			/* Too new? Too old? OpenSSL docs are the worst!
		case SSL_ERROR_WANT_ACCEPT:
			fprintf( stderr, "SSL_ERROR_WANT_ACCEPT\n" );
			return false;
			*/ 
		case SSL_ERROR_WANT_X509_LOOKUP:
			fprintf(stderr, "SSL_X509_LOOKUP\n");
			break;
			
		case SSL_ERROR_SYSCALL:
			fprintf(stderr, "SSL_ERROR_SYSCALL\n");
			break;
			
		case SSL_ERROR_SSL:
			fprintf(stderr, "SSL_ERROR_SSL\n");
			break;
			
		default:
			fprintf(stderr, "SSL Error unknown!\n");
			break;
#endif
	}

	/* Failure of some kind */
	SSL_free(*ssl);

	return false;
}

int sslRecv(SSL *ssl, char *buffer, unsigned int maxlength) {
	int status;

	status = SSL_read(ssl, buffer, maxlength);
	if (status < 0) {
		BIO_printf(stderrBio, "%s\n", "SSL read failed.");
		ERR_print_errors(stderrBio);
	}

	return status;
}

int sslSend(SSL *ssl, char *buffer, unsigned int length) {
	int status;
	
	status = SSL_write(ssl, buffer, length);
	if (status < 0) {
		BIO_printf(stderrBio, "%s\n", "SSL write failed.");
		ERR_print_errors(stderrBio);
	}

	return status;
}

int sslClose(SSL *ssl) {
	return SSL_shutdown(ssl);
}
#endif
