#include "config.h"
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <time.h>
#include <pthread.h>
#include <sys/file.h>
#include "log.h"

/* Open a logfile.
 */
FILE *open_logfile(char *logfile) {
	FILE *fp;

	if ((fp = fopen(logfile, "a")) != NULL) {
		flock(fileno(fp), LOCK_EX);
	}

	return fp;
}

void close_logfile(FILE *fp) {
	flock(fileno(fp), LOCK_UN);
	fclose(fp);
}

/* Write a timestamp to a logfile.
 */
static void print_timestamp(FILE *fp) {
	time_t t;
	struct tm *s;
	char timestr[51];

	time(&t);
	s = localtime(&t);
	*(timestr + 50) = '\0';
	strftime(timestr, 50, "%a %d %b %Y %X", s);
	fprintf(fp, "%s|", timestr);
}

/* Write an IP address to a logfile.
 */
static void print_ip_address(FILE *fp, long ip_address) {
	unsigned char *ip;

	ip = (unsigned char*)&ip_address;
	fprintf(fp, "%hhu.%hhu.%hhu.%hhu|", ip[0], ip[1], ip[2], ip[3]);
}

char *secure_string(char *str) {
	char *c = str;

	while (*c != '\0') {
		if (*c == '\27') {
			*c = ' ';
		}
		c++;
	}

	return str;
}

// ---< Main log functions >-------------

/* Log a text.
 */
void log_string(char *logfile, char *data) {
	FILE *fp;

	if ((fp = open_logfile(logfile)) != NULL) {
		print_timestamp(fp);
		fprintf(fp, "%s\n", data);
		close_logfile(fp);
	}
}

/* Log a CGI error.
 */
void log_CGI_error(char *logfile, char *CGI_file, char *error) {
	FILE *fp;
	char *c;

	c = error;
	while (*c != '\0') {
		if (*c == '\n') {
			*c = '|';
		}
		c++;
	}

	if ((fp = open_logfile(logfile)) != NULL) {
		print_timestamp(fp);
		fprintf(fp, "%s|%s\n", CGI_file, secure_string(error));
		close_logfile(fp);
	}
}

/* Log a warning.
 */
void log_warning(char *logfile, char *data, long ip_address) {
	FILE *fp;

	if ((fp = open_logfile(logfile)) != NULL) {
		print_ip_address(fp, ip_address);
		print_timestamp(fp);
		fprintf(fp, "%s\n", data);
		close_logfile(fp);
	}
}

/* Log a HTTP request.
 */
void log_request(t_session *session, int code) {
	FILE *fp;
	char rcode[4], *user;
	t_headerfield *headerfield;
	bool should_log;

	if (ip_allowed(session->ip_address, session->config->LogfileMask) != deny) {
		sprintf(rcode, "%d", code);

		if ((fp = open_logfile(session->host->LogFile)) != NULL) {
			print_ip_address(fp, session->ip_address);
			print_timestamp(fp);
			if ((user = session->remote_user) == NULL) {
				user = "";
			}
			fprintf(fp, "%s|%s|%s %s", rcode, secure_string(user), secure_string(session->method), secure_string(session->URI));
			if (session->vars != NULL) {
				fprintf(fp, "?%s", secure_string(session->vars));
			}
			fprintf(fp, " %s", secure_string(session->http_version));
			
			headerfield = session->headerfields;
			while (headerfield != NULL) {
				should_log = true;
				if (strlen(headerfield->data) >= 14) {
					if (memcmp("Authorization:", headerfield->data, 14) == 0) {
						should_log = false;
					}
				}
				if (should_log) {
					fprintf(fp, "|%s", secure_string(headerfield->data));
				}
				headerfield = headerfield->next;
			}
			fprintf(fp, "\n");

			close_logfile(fp);
		}
	}
}

/* Log garbage sent by a client.
 */
void log_garbage(t_session *session) {
	FILE *fp;

	if ((session->config->GarbageLogfile != NULL) && (session->request != NULL)) {
		if ((fp = open_logfile(session->config->GarbageLogfile)) != NULL) {
			print_ip_address(fp, session->ip_address);
			print_timestamp(fp);
			fprintf(fp, "%s\n", session->request);
			close_logfile(fp);
		}
	}
}

/* Log an unbanning.
 */
void log_unban(char *logfile, unsigned long ip_address, unsigned long connect_attempts) {
	FILE *fp;

	if ((fp = open_logfile(logfile)) != NULL) {
		print_ip_address(fp, ip_address);
		print_timestamp(fp);
		fprintf(fp, "Unbanned (%ld connect attempts during ban)\n", connect_attempts);
		close_logfile(fp);
	}
}

#ifdef DEBUG
void log_error(char *logfile, int code) {
	FILE *fp;
	char mesg[256];

	if ((fp = open_logfile(logfile)) != NULL) {
		print_timestamp(fp);
		mesg[255] = 0;
		strerror_r(code, mesg, 255);
		fprintf(fp, "ERROR: %s\n", mesg);
		close_logfile(fp);
	}
}
#endif
