/* Hiawatha | send.c
 * 
 * All the routines that send data directly to the client.
 */
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <errno.h>
#ifdef HAVE_SSL
#include "libssl.h"
#endif
#include "send.h"
#include "session.h"
#include "libstr.h"

#define MAX_CHUNK_SIZE 2048

char *hs_http   = "HTTP/1.1 ";                  //  9
char *hs_server = "Server: ";                   //  8
char *hs_conn   = "Connection: ";               // 12
char *hs_concl  = "close\r\n";                  //  7
char *hs_conka  = "keep-alive\r\n";             // 12
char *hs_contyp = "Content-Type: ";             // 14
char *hs_lctn   = "Location: http";             // 14
char *hs_slash  = "://";                        //  3
char *hs_range  = "Accept-Ranges: bytes\r\n";   // 22
char *hs_gzip   = "Content-Encoding: gzip\r\n"; // 24
char *hs_eol    = "\r\n";                       //  2

/* Send a char buffer to the client. Traffic throttling is handled here.
 */
int send_buffer(t_session *session, const char *buffer, int size) {
	int bytes_send = 0, total_send = 0, can_send, rest;
	time_t new_time;

	/* Send buffer to browser.
	 */
	if (size > 0) {
		if (session->Directory != NULL) {
			if (session->Directory->SessionSpeed > 0) {
				session->throttle = session->Directory->SessionSpeed;
			}
		}
		do {
			rest = size - total_send;
			if (session->throttle > 0) {
				do {
					new_time = time(NULL);
					if (session->throttle_timer < new_time) {
						session->bytecounter = 0;
						session->throttle_timer = new_time;
					}
					can_send = session->throttle - session->bytecounter;
					if (can_send <= 0) {
						usleep(10000);
					}
				} while (can_send <= 0);
				if (can_send > rest) {
					can_send = rest;
				}
			} else {
				can_send = rest;
			}
			
#ifdef HAVE_SSL
			if (session->via_ssl) {
				if ((bytes_send = sslSend(session->ssl_data, (char*)(buffer + total_send), can_send)) <= 0) {
					bytes_send = -1;
				}
			} else
#endif
				bytes_send = send(session->client_socket, buffer + total_send, can_send, 0);

			if (bytes_send == -1) {
				close_socket(session);
				session->keep_alive = false;
				return -1;
			}
			total_send += bytes_send;
			session->bytecounter += bytes_send;
		} while (total_send < size);
	}

	return 0;
}

/* Send a HTTP header to the client. Header is not closed by this function.
 */
int send_header(t_session *session, int code) {
	int retval = -1;
	char ecode[4], timestr[64];
	const char *emesg;
	time_t t;
	struct tm *s;

	time(&t);
	s = gmtime(&t);
	strftime(timestr, 64, "%a, %d %b %Y %X GMT\r\n", s);

	/* Send HTTP header.
	 */
	sprintf(ecode, "%d", code);
	emesg = errorstr(code);
	do {
		if (send_buffer(session, hs_http, 9) == -1) {
			break;
		} else if (send_buffer(session, ecode, 3) == -1) {
			break;
		} else if (send_buffer(session, " ", 1) == -1) {
			break;
		} else if (send_buffer(session, emesg, strlen(emesg)) == -1) {
			break;
		} else if (send_buffer(session, hs_eol, 2) == -1) {
			break;
		} else if (send_buffer(session, "Date: ", 6) == -1) {
			break;
		} else if (send_buffer(session, timestr, strlen(timestr)) == -1) {
			break;
		} else if (send_buffer(session, hs_server, 8) == -1) {
			break;
		} else if (send_buffer(session, session->config->ServerString, strlen(session->config->ServerString)) == -1) {
			break;
		} else if (send_buffer(session, hs_eol, 2) == -1) {
			break;
		} else if ((session->is_CGI_script == false) && (session->URI_is_dir == false)) {
			if (send_buffer(session, hs_range, 22) == -1) {
				break;
			}
		}
		if (send_buffer(session, hs_conn, 12) == -1) {
			break;
		} else if (session->keep_alive) {
			if (send_buffer(session, hs_conka, 12) == -1) {
				break;
			}
		} else if (send_buffer(session, hs_concl, 7) == -1) {
			break;
		}
		if (session->encode_gzip) {
			if (send_buffer(session, hs_gzip, 24) == -1) {
				break;
			}
		}

		if (session->mimetype != NULL) {
			if (send_buffer(session, hs_contyp, 14) == -1) {
				break;
			} else if (send_buffer(session, session->mimetype, strlen(session->mimetype)) == -1) {
				break;
			} else if (send_buffer(session, "\r\n", 2) == -1) {
				break;
			}
		}

		retval = 0;
	} while (false);

	return retval;
}

/* Send a datachunk to the client, used by run_script() in target.c
 */
int send_chunk(t_session *session, char *chunk, int size) {
	char hex[8];
	bool must_send = false, copy_chunk = false, last_chunk = false;

	if (chunk != NULL) {
		if (session->chunksize + size <= MAX_CHUNK_SIZE) {
			if (session->chunkbuffer != NULL) {
				memcpy(session->chunkbuffer + session->chunksize, chunk, size);
				session->chunksize += size;
			} else {
				copy_chunk = true;
			}
		} else {
			must_send = true;
			copy_chunk = true;
		}
	} else {
		if (session->HEAD_request) {
			if (session->chunkbuffer != NULL) {
				free(session->chunkbuffer);
				session->chunkbuffer = NULL;
				session->chunksize = 0;
			}
		} else {
			must_send = true;
			last_chunk = true;
		}
	}

	if (must_send && (session->chunksize > 0)) {
		if (session->keep_alive) {
			if (sprintf(hex, "%x\r\n", session->chunksize) >= 0) {
				if (send_buffer(session, hex, strlen(hex)) == -1) {
					return -1;
				}
			} else {
				session->keep_alive = false;
			}
		}
		if (send_buffer(session, session->chunkbuffer, session->chunksize) == -1) {
			return -1;
		} else if (session->keep_alive) {
			if (send_buffer(session, "\r\n", 2) == -1) {
				return -1;
			}
		}
		free(session->chunkbuffer);
		session->chunkbuffer = NULL;
		session->chunksize = 0;
	}

	if (last_chunk && session->keep_alive) {
		send_buffer(session, "0\r\n\r\n", 5);
	}

	if (copy_chunk) {
		if ((session->chunkbuffer = (char*)malloc(MAX_CHUNK_SIZE)) != NULL) {
			memcpy(session->chunkbuffer, chunk, size);
			session->chunksize = size;
		} else {
			return -1;
		}
	}

	return 0;
}

/* Send a HTTP code to the client. Used in case of an error.
 */
int send_code(t_session *session, int code) {
	int retval = -1, binded_port;
	char ecode[4], len[8], port[16];
	const char *emesg;

	/* Send simple HTTP error message.
	 */
	if (send_header(session, code) == -1) {
		return retval;
	}

	switch (code) {
		case 301:
			if (send_buffer(session, hs_lctn, 14) == -1) {
				break;
			}
#ifdef HAVE_SSL
			if (session->via_ssl) {
				if (send_buffer(session, "s", 1) == -1) {
					break;
				}
			}
#endif
			if (send_buffer(session, hs_slash, 3) == -1) {
				break;
			}

			if (send_buffer(session, *(session->host->Hostname.item), strlen(*(session->host->Hostname.item))) == -1) {
				break;
			}

#ifdef HAVE_SSL
			if (session->via_ssl) {
				binded_port = 443;
			} else 
#endif
				binded_port = 80;

			if (session->binding->port != binded_port) {
				sprintf(port, ":%d", session->binding->port);
				if (send_buffer(session, port, strlen(port)) == -1) {
					break;
				}
			}
			if (send_buffer(session, session->URI, strlen(session->URI)) == -1) {
				break;
			} else if (send_buffer(session, "/", 1) == -1) {
				break;
			}
			if (session->vars != NULL) {
				if (send_buffer(session, "?", 1) == -1) {
					break;
				} else if (send_buffer(session, session->vars, strlen(session->vars)) == -1) {
					break;
				}
			}
			if (send_buffer(session, "\r\n", 2) == -1) {
				break;
			}
			retval = 0;
			break;
		case 401:
			send_basic_auth(session);
			retval = 0;
			break;
		default:
			retval = 0;
			break;
	}

	if (retval == 0) {
		do {
			retval = -1;
			sprintf(ecode, "%d", code);
			emesg = errorstr(code);
			sprintf(len, "%d", 2 * strlen(emesg) + 353);

			if (send_buffer(session, "Content-Length: ", 16) == -1) {
				break;
			} else if (send_buffer(session, len, strlen(len)) == -1) {
				break;
			} else if (send_buffer(session, "\r\n", 2) == -1) {
				break;
			} else if (send_buffer(session, hs_contyp, 14) == -1) {
				break;
			} else if (send_buffer(session, "text/html\r\n\r\n", 13) == -1) {
				break;
			} else if (session->HEAD_request) {
				retval = 0;
				break;
			} else if (send_buffer(session, "<html><head><title>", 19) == -1) {
				break;
			} else if (send_buffer(session, ecode, 3) == -1) {
				break;
			} else if (send_buffer(session, ": ", 2) == -1) {
				break;
			} else if (send_buffer(session, emesg, strlen(emesg)) == -1) {
				break;
			} else if (send_buffer(session,	"</title>\n" // 9
											"<style type=\"text/css\">" // 23
											"BODY { color:#ffff00 ; background-color:#000040 } " // 50
											"H1 { font-size:50px ; margin:0px ; font-weight:normal } " // 56
											"TD { font-family: arial ; font-size:20px ; text-align:center ; vertical-align:middle }" //86
											"</style></head>\n" // 16
											"<body><table width=100% height=100%><tr><td><h1>", 288) == -1) { // 48
				break;
			} else if (send_buffer(session, ecode, 3) == -1) {
				break;
			} else if (send_buffer(session, "</h1>", 5) == -1) {
				break;
			} else if (send_buffer(session, emesg, strlen(emesg)) == -1) {
				break;
			} else if (send_buffer(session, "</td></tr></table></body></html>\n", 33) == -1) {
				break;
			}

			retval = 0;
		} while (false);
	}

	return retval;
}

/* Send a Basic Authentication message to the client.
 */
int send_basic_auth(t_session *session) {
	int retval = -1;

	do {
		if (send_buffer(session, "WWW-Authenticate: Basic", 23) == -1) {
			break;
		} else if (session->host->LoginMessage != NULL) {
			if (send_buffer(session, " realm=\"", 8) == -1) {
				break;
			} else if (send_buffer(session, session->host->LoginMessage, strlen(session->host->LoginMessage)) == -1) {
				break;
			} else if (send_buffer(session, "\"", 1) == -1) {
				break;
			}
		}
		if (send_buffer(session, "\r\n", 2) == -1) {
			break;
		}
		retval = 0;
	} while (false);

	return retval;
}
