#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include "serverconfig.h"
#include "libstr.h"

#define mc_INCLUDE      -1
#define mc_READ_ERROR   -2
#define mc_DIR_CONFLICT -3
#define mc_ID_CONFLICT  -4
#define mc_NO_HOSTNAME  -5
#define mc_NO_PORT      -6
#define mc_REQUIRE      -7

#define id_NOBODY (uid_t)65534

bool including = false;
t_keyvalue *variables = NULL;

static t_host *new_host(void) {
	t_host *host;

	if ((host = (t_host*)malloc(sizeof(t_host))) != NULL) {
		host->WebsiteRoot       = WEBROOT_DIR;
		host->StartFile         = "index.html";
		host->ErrorHandler      = NULL;
		host->ErrorCode         = -1;
		host->LogFile           = LOG_DIR"/access.log";
		init_charlist(&(host->Hostname));
		host->UserWebsites      = false;
		host->ExeCGI            = false;
		host->ShowIndex         = false;
		host->AccessList        = NULL;
#ifdef HAVE_PLUGIN
		host->PluginActive      = false;
#endif
		host->LoginMessage      = "Private page";
		host->PasswordFile      = NULL;
		host->GroupFile         = NULL;
		host->DenyBot           = NULL;
		init_charlist(&(host->RequiredBinding));
		init_charlist(&(host->RequiredGroup));
		host->UserId            = -1;
		host->GroupId           = -1;
		host->Groups.number     = 0;
		host->Groups.array      = NULL;
		init_charlist(&(host->Volatile));
		init_charlist(&(host->ImageReferer));
		host->ImgRefReplacement = NULL;
		host->EnvirStr          = NULL;
		host->Alias             = NULL;
#ifdef HAVE_SSL
		host->RequireSSL        = false;
#endif
		host->PreventXSS        = false;
		host->PreventSQLi       = false;
		host->FollowSymlinks    = false;
		host->next              = NULL;
	}

	return host;
}

static t_directory *new_directory(void) {
	t_directory *directory;

	if ((directory = (t_directory*)malloc(sizeof(t_directory))) != NULL) {
		directory->Path              = NULL;
		directory->UserId            = -1;
		directory->GroupId           = -1;
		directory->ExeCGIset         = false;
		directory->ShowIndexSet      = false;
		directory->FollowSymlinksSet = false;
		directory->AccessList        = NULL;
		directory->Groups.number     = 0;
		directory->Groups.array      = NULL;
		directory->PasswordFile      = NULL;
		directory->GroupFile         = NULL;
		init_charlist(&(directory->RequiredGroup));
		init_charlist(&(directory->ImageReferer));
		directory->ImgRefReplacement = NULL;
		directory->MaxClients        = -1;
		directory->Clients           = 0;
		directory->UploadSpeed       = 0;
		directory->SessionSpeed      = 0;
		directory->EnvirStr          = NULL;
		pthread_mutex_init(&(directory->client_mutex), NULL);
		directory->next              = NULL;
	}

	return directory;
}

t_config *default_config(void) {
	t_config *config;

	if ((config = (t_config*)malloc(sizeof(t_config))) != NULL) {
		config->ServerRoot       = NULL;
		config->ServerUid        = id_NOBODY;
		config->ServerGid        = id_NOBODY;
		config->ServerString     = "Hiawatha v"VERSION;
		config->MaxReqSize       = 65536;
		config->Groups.number    = 0;
		config->Groups.array     = NULL;
		config->BindHTTP         = NULL;
#ifdef HAVE_SSL
		config->BindHTTPS        = NULL;
		config->ServerKey        = CONFIG_DIR"/serverkey.pem";
#endif
		init_charlist(&(config->CGIextension));
		config->ConnectionsTotal = 50;
		config->ConnectionsPerIP = 10;
		config->TimeForCGI       = 5;
		config->TimeForRequest   = 30;
		config->MaxKeepAlive     = 50;
		config->EnableTRACE       = false;
		config->first_host       = new_host();
		config->HomedirSource    = hiaconf;
#ifdef HAVE_PLUGIN
		config->PluginURI        = NULL;
#endif
		config->Mimetype         = NULL;
		config->Directory        = NULL;
		config->Throttle         = NULL;
		config->Homedir          = NULL;
		config->CGIhandler       = NULL;

		config->SystemLogfile    = LOG_DIR"/system.log";
		config->GarbageLogfile   = NULL;
		config->LogfileMask       = NULL;

		config->BanOnGarbage     = 0;
		config->BanOnMaxPerIP    = 2;
		config->BanOnFlooding    = 0;
		config->BanOnMaxReq      = 0;
		config->BanOnSQLi        = 0;
		config->KickOnBan        = false;
		config->RebanDuringBan   = false;
		config->FloodingCount    = 0;
		config->FloodingTime     = 0;
		config->BanlistMask      = NULL;

#ifdef HAVE_COMMAND
		config->CommandPort      = NULL;
		config->CommandPassword  = "*";
#endif
	}

	return config;
}

static bool valid_file(char *file) {
	bool retval = false;

	if (file != NULL) {
		if (strchr(file, '/') == NULL) {
			if (strlen(file) < 32) {
				retval = true;
			}
		}
	}

	return retval;
}

static bool valid_directory(char *dir) {
	bool retval = false;
	int len;

	if (dir != NULL) {
		if (*dir == '/') {
			if ((len = strlen(dir)) < 128) {
				if (*(dir + len - 1) != '/') {
					retval = true;
				}
			}
		}
	}

	return retval;
}

int check_configuration(t_config *config) {
	t_host *host;
	t_directory *dir;

	if (config->first_host->Hostname.size == 0) {
		return mc_NO_HOSTNAME;
	}

	if ((config->BindHTTP == NULL)
#ifdef HAVE_SSL
		&& (config->BindHTTPS == NULL)
#endif
		) {
		return mc_NO_PORT;
	}

	host = config->first_host;
	while (host != NULL) {
		if (config->ServerRoot != NULL) {
			if (host->UserWebsites) {
				return mc_DIR_CONFLICT;
			}
		}
		if (config->ServerUid == (uid_t)-1) {
			if (host->UserId == (uid_t)-1) {
				host->UserId = host->GroupId = id_NOBODY;
			}
		} else {
			if (host->UserId != (uid_t)-1) {
				return mc_ID_CONFLICT;
			}
		}
		host = host->next;
	}

	if (config->ServerUid != (uid_t)-1) {
		dir = config->Directory;
		while (dir != NULL) {
			if (dir->UserId != (uid_t)-1) {
				return mc_ID_CONFLICT;
			}
			dir = dir->next;
		}
	}

#ifdef HAVE_SSL
	if ((config->first_host->RequireSSL == true) || (config->first_host->RequiredBinding.size > 0)) {
		return mc_REQUIRE;
	}
#endif

	return 0;
}

static int parse_yesno(char *yesno, bool *result) {
	if ((strcmp(yesno, "yes") == 0) || (strcmp(yesno, "true") == 0)) {
		*result = true;
	} else if ((strcmp(yesno, "no") == 0) || (strcmp(yesno, "false") == 0)) {
		*result = false;
	} else {
		return -1;
	}

	return 0;
}

static gid_t parse_groups(char *gid, t_groups *groups) {
	gid_t retval, *id;
	int i;
	char *c;

	c = gid;
	while (*c != '\0') {
		if (*c == ',') {
			(*groups).number++;
			*c = '\0';
		}
		c++;
	}
	if ((retval = (gid_t)str2int(gid)) > (gid_t)0) {
		if ((*groups).number > 0) {
			(*groups).array = id = (gid_t*)malloc((*groups).number * sizeof(int));
			for (i = 0; i < (*groups).number; i++) {
				gid = gid + strlen(gid) + 1;
				if ((*id = (gid_t)str2int(gid)) > (gid_t)0) {
					id++;
				} else {
					retval = (gid_t)-1;
					break;
				}
			}
		}
	} else {
		retval = (gid_t)-1;
	}

	return retval;
}

int parse_passwordfile(char *line, char **pwdfile) {
	if (*pwdfile != NULL) {
		free(*pwdfile);
		*pwdfile = NULL;
	}
	if (strcmp(line, "none") != 0) {
		if (strlen(line) >= 256) {
			return -1;
		} else if ((*pwdfile = strdup(line)) == NULL) {
			return -1;
		}
	}

	return 0;
}
static bool was_system_setting(char *key, char *value, t_config *config) {
	bool retval = true;
	char *uid, *gid, *rest;
	t_CGIhandler *CGI;
#ifdef HAVE_COMMAND
	char *port, *password;
#endif

	do {
		if (strcmp(key, "banlistmask") == 0) {
			if ((config->BanlistMask = parse_accesslist(value, false, config->BanlistMask)) != NULL) {
				break;
			}
		} else if (strcmp(key, "banonflooding") == 0) {
			if (split_string(value, &value, &rest, '/') == -1) {
			} else if ((config->FloodingCount = str2int(value)) <= 0) {
			} else if (split_string(rest, &value, &rest, ':') != 0) {
			} else if ((config->FloodingTime = str2int(value)) <= 0) {
			} else if ((config->BanOnFlooding = str2int(rest)) > 0) {
				config->FloodingTime--;
				break;
			}
		} else if (strcmp(key, "banongarbage") == 0) {
			if ((config->BanOnGarbage = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonmaxperip") == 0) {
			if ((config->BanOnMaxPerIP = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonmaxreqsize") == 0) {
			if ((config->BanOnMaxReq = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonsqli") == 0) {
			if ((config->BanOnSQLi = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "bindhttp") == 0) { 
			if ((config->BindHTTP = parse_bindlist(value, 80, config->BindHTTP)) != NULL) {
				break;
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "bindhttps") == 0) { 
			if ((config->BindHTTPS = parse_bindlist(value, 443, config->BindHTTPS)) != NULL) {
				break;
			}
#endif
#ifdef HAVE_COMMAND
		} else if (strcmp(key, "commandchannel") == 0) {
			if (split_string(value, &port, &password, ',') == 0) {
				config->CommandPort = (t_bindlist*)malloc(sizeof(t_bindlist));
				config->CommandPort->next = NULL;
				config->CommandPort->ip = htonl(0x7F000001);
				config->CommandPort->socket = -1;
				config->CommandPort->name = NULL;
				if ((config->CommandPort->port = str2int(port)) > 0) {
					if ((config->CommandPassword = strdup(password)) != NULL) {
						break;
					}
				} else {
					free(config->CommandPort);
				}
			}
#endif
		} else if (strcmp(key, "cgiextension") == 0) {
			if (parse_charlist(value, &(config->CGIextension)) != -1) {
				break;
			}
		} else if (strcmp(key, "cgihandler") == 0) {
			if (split_string(value, &value, &rest, ':') == 0) {
				CGI = config->CGIhandler;
				config->CGIhandler = (t_CGIhandler*)malloc(sizeof(t_CGIhandler));
				config->CGIhandler->next = CGI;
				config->CGIhandler->handler = strdup(value);
				init_charlist(&(config->CGIhandler->extension));
				if (parse_charlist(rest, &(config->CGIhandler->extension)) != -1) {
					break;
				}
			}
		} else if (strcmp(key, "enabletrace") == 0) {
			if (parse_yesno(value, &(config->EnableTRACE)) == 0) {
				break;
			}
		} else if (strcmp(key, "homedirsource") == 0) {
			if (strcmp(value, "hiawatha") == 0) {
				config->HomedirSource = hiaconf;
				break;
			} else if (strcmp(value, "system") == 0) {
				config->HomedirSource = passwd;
				break;
			}
		} else if (strcmp(key, "connectionsperip") == 0) {
			if ((config->ConnectionsPerIP = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "connectionstotal") == 0) {
			if ((config->ConnectionsTotal = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "garbagelogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((config->GarbageLogfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "kickonban") == 0) {
			if (parse_yesno(value, &(config->KickOnBan)) == 0) {
				break;
			}
		} else if (strcmp(key, "rebanduringban") == 0) {
			if (parse_yesno(value, &(config->RebanDuringBan)) == 0) {
				break;
			}
		} else if (strcmp(key, "logfilemask") == 0) {
			if ((config->LogfileMask = parse_accesslist(value, false, config->LogfileMask)) != NULL) {
				break;
			}
		} else if (strcmp(key, "maxkeepalive") == 0) {
			if ((config->MaxKeepAlive = str2int(value)) != -1) {
				break;
			}
#ifdef HAVE_PLUGIN
		} else if (strcmp(key, "pluginuri") == 0) {
			if (strlen(value) < 16) {
				if ((config->PluginURI = strdup(value)) != NULL) {
					break;
				}
			}
#endif
		} else if (strcmp(key, "maxrequestsize") == 0) {
			if ((config->MaxReqSize = str2int(value)) > 0) {
				config->MaxReqSize <<= 10;
				break;
			}
		} else if (strcmp(key, "serverid") == 0) {
			if (strcmp(value, "perdomain") == 0) {
				config->ServerUid = (uid_t)-1;
				config->ServerGid = (gid_t)-1;
				break;
			} else if (split_string(value, &uid, &gid, ':') == 0) {
				if ((config->ServerUid = (uid_t)str2int(uid)) > (uid_t)0) {
					if ((config->ServerGid = parse_groups(gid, &(config->Groups))) != (gid_t)-1) {
						break;
					}
				}
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "serverkey") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((config->ServerKey = strdup(value)) != NULL) {
					break;
				}
			}
#endif
		} else if (strcmp(key, "serverroot") == 0) {
			if (valid_directory(value)) {
				if ((config->ServerRoot = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "serverstring") == 0) {
			if (strlen(value) < 128) {
				if ((config->ServerString = strdup(remove_spaces(value))) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "systemlogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((config->SystemLogfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "timeforcgi") == 0) {
			if ((config->TimeForCGI = str2int(value)) >= 1) {
				break;
			}
		} else if (strcmp(key, "timeforrequest") == 0) {
			if ((config->TimeForRequest = str2int(value)) >= 1) {
				break;
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool was_config_setting(char *key, char *value, t_host *host) {
	bool retval = true;
	char *gid, *code;
	int result;

	do {
		if (strcmp(key, "accesslist") == 0) {
			if ((host->AccessList = parse_accesslist(value, true, host->AccessList)) != NULL) {
				break;
			}
		} else if (strcmp(key, "errorhandler") == 0) {
			result = split_string(value, &value, &code, ',');
			if ((strlen(value) < 128) && (*value == '/')) {
				if ((host->ErrorHandler = strdup(value)) != NULL) {
					if (result == 0) {
						if ((host->ErrorCode = str2int(code)) != -1) {
							break;
						}
					} else {
						break;
					}
				}
			}
		} else if (strcmp(key, "loginmessage") == 0) {
			if (strlen(value) < 64) {
				if ((host->LoginMessage = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "passwordfile") == 0) {
			if (parse_passwordfile(value, &(host->PasswordFile)) == 0) {
				break;
			}
		} else if (strcmp(key, "requiredgroup") == 0) {
			if (split_string(value, &gid, &value, ':') == 0) {
				if (strlen(gid) < 32) {
					if (parse_charlist(gid, &(host->RequiredGroup)) != -1) {
						if (parse_passwordfile(value, &(host->GroupFile)) == 0) {
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "setenvir") == 0) {
			if ((host->EnvirStr = parse_keyvaluelist(value, host->EnvirStr, ':')) != NULL) {
				break;
			}
		} else if (strcmp(key, "showindex") == 0) {
			if (parse_yesno(value, &(host->ShowIndex)) == 0) {
				break;
			}
		} else if (strcmp(key, "startfile") == 0) {
			if (valid_file(value)) {
				if ((host->StartFile = strdup(value)) != NULL) {
					break;
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool was_host_setting(char *key, char *value, t_host *host) {
	bool retval = true;
	char *uid, *gid, *botname;
	t_denybotlist *denybot;
	t_keyvalue *alias;

	do {
		if (strcmp(key, "alias") == 0) {
			if ((host->Alias = parse_keyvaluelist(value, host->Alias, ':')) != NULL) {
				alias = host->Alias;
				while (alias != NULL) {
					if ((valid_directory(alias->key) == false) || (valid_directory(alias->value) == false)) {
						retval = false;
						break;
					}
					alias = alias->next;
				}
				break;
			}
		} else if (strcmp(key, "denybot") == 0) {
			if (split_string(value, &botname, &value, ':') == 0) {
				denybot = (t_denybotlist*)malloc(sizeof(t_denybotlist));
				denybot->next = host->DenyBot;
				host->DenyBot = denybot;

				init_charlist(&(denybot->uri));
				denybot->bot = strdup(botname);
				if (parse_charlist(value, &(denybot->uri)) == 0) {
					break;
				}
			}
		} else if (strcmp(key, "executecgi") == 0) {
			if (parse_yesno(value, &(host->ExeCGI)) == 0) {
				break;
			}
		} else if (strcmp(key, "followsymlinks") == 0) {
			if (parse_yesno(value, &(host->FollowSymlinks)) == 0) {
				break;
			}
		} else if (strcmp(key, "hostid") == 0) {
			if (split_string(value, &uid, &gid, ':') == 0) {
				if ((host->UserId = (uid_t)str2int(uid)) > 0) {
					if ((host->GroupId = parse_groups(gid, &(host->Groups))) != (gid_t)-1) {
						break;
					}
				}
			}
		} else if (strcmp(key, "hostname") == 0) {
			if (parse_charlist(value, &(host->Hostname)) == 0) {
				break;
			}
		} else if (strcmp(key, "imagereferer") == 0) {
			if (split_string(value, &value, &(host->ImgRefReplacement), ':') == 0) {
				if ((host->ImgRefReplacement = strdup(host->ImgRefReplacement)) != NULL) {
					if (parse_charlist(value, &(host->ImageReferer)) == 0) {
						break;
					}
				}
			}
		} else if (strcmp(key, "logfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((host->LogFile = strdup(value)) != NULL) {
					break;
				}
			}
#ifdef HAVE_PLUGIN
		} else if (strcmp(key, "pluginactive") == 0) {
			if (parse_yesno(value, &(host->PluginActive)) == 0) {
				break;
			}
#endif
		} else if (strcmp(key, "preventsqli") == 0) {
			if (parse_yesno(value, &(host->PreventSQLi)) == 0) {
				break;
			}
		} else if (strcmp(key, "preventxss") == 0) {
			if (parse_yesno(value, &(host->PreventXSS)) == 0) {
				break;
			}
		} else if ((strcmp(key, "requiredbinding") == 0) || (strcmp(key, "requirebinding") == 0)) {
			if (parse_charlist(value, &(host->RequiredBinding)) == 0) {
				break;
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "requiressl") == 0) {
			if (parse_yesno(value, &(host->RequireSSL)) == 0) {
				break;
			}
#endif
		} else if (strcmp(key, "userwebsites") == 0) {
			if (parse_yesno(value, &(host->UserWebsites)) == 0) {
				break;
			}
		} else if (strcmp(key, "volatileobject") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				host->Volatile.size++;
				if ((host->Volatile.item = (char**)realloc(host->Volatile.item, host->Volatile.size * sizeof(char*))) != NULL) {
					if ((*(host->Volatile.item + host->Volatile.size - 1) = strdup(value)) != NULL) {
						break;
					}
				}
			}
		} else if (strcmp(key, "websiteroot") == 0) {
			if (valid_directory(value)) {
				if ((host->WebsiteRoot = strdup(value)) != NULL) {
					break;
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool was_directory_setting(char *key, char *value, t_directory *directory) {
	bool retval = true;
	char *uid, *gid, *maxclients;
	int length;

	do {
		if (strcmp(key, "accesslist") == 0) {
			if ((directory->AccessList = parse_accesslist(value, true, directory->AccessList)) != NULL) {
				break;
			}
		} else if (strcmp(key, "executecgi") == 0) {
			if (parse_yesno(value, &(directory->ExeCGI)) == 0) {
				directory->ExeCGIset = true;
				break;
			}
		} else if (strcmp(key, "followsymlink") == 0) {
			if (parse_yesno(value, &(directory->FollowSymlinks)) == 0) {
				directory->FollowSymlinksSet = true;
				break;
			}
		} else if (strcmp(key, "hostid") == 0) {
			if (split_string(value, &uid, &gid, ':') == 0) {
				if ((directory->UserId = (uid_t)str2int(uid)) > 0) {
					if ((directory->GroupId = parse_groups(gid, &(directory->Groups))) != (gid_t)-1) {
						break;
					}
				}
			}
		} else if (strcmp(key, "imagereferer") == 0) {
			if (split_string(value, &value, &(directory->ImgRefReplacement), ':') == 0) {
				if ((directory->ImgRefReplacement = strdup(directory->ImgRefReplacement)) != NULL) {
					if (parse_charlist(value, &(directory->ImageReferer)) == 0) {
						break;
					}
				}
			}
		} else if (strcmp(key, "passwordfile") == 0) {
			if (parse_passwordfile(value, &(directory->PasswordFile)) == 0) {
				break;
			}
		} else if (strcmp(key, "path") == 0) {
			length = strlen(value);
			if ((length < 128) && (*value == '/')) {
				if (*(value + length - 1) == '/') {
					if (length >= 3) {
						directory->PathMatch = part;
						if ((directory->Path = strdup(value)) != NULL) {
							break;
						}
					}
				} else {
					if (length >= 2) {
						directory->PathMatch = root;
						if ((directory->Path = (char*)malloc(length + 2)) != NULL) {
							memcpy(directory->Path, value, length);
							memcpy(directory->Path + length, "/\0", 2);
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "requiredgroup") == 0) {
			if (split_string(value, &gid, &value, ':') == 0) {
				if (strlen(gid) < 32) {
					if (parse_charlist(gid, &(directory->RequiredGroup)) != -1) {
						if (parse_passwordfile(value, &(directory->GroupFile)) == 0) {
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "setenvir") == 0) {
			if ((directory->EnvirStr = parse_keyvaluelist(value, directory->EnvirStr, 'c')) != NULL) {
				break;
			}
		} else if (strcmp(key, "showindex") == 0) {
			if (parse_yesno(value, &(directory->ShowIndex)) == 0) {
				directory->ShowIndexSet = true;
				break;
			}
		} else if (strcmp(key, "uploadspeed") == 0) {
			if (split_string(value, &value, &maxclients, ',') == 0) {
				if ((directory->UploadSpeed = str2int(value)) > 0) {
					directory->UploadSpeed <<= 10;
					if ((directory->MaxClients = str2int(maxclients)) > 0) {
						break;
					}
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

void print_configerror(char *configfile, int code) {
	if (code != mc_INCLUDE) {
		printf("\n[%s] Fatal error: ", configfile);
		switch (code) {
			case mc_READ_ERROR:
				printf("can't read configfile %s.\n", configfile);
				break;
			case mc_DIR_CONFLICT:
				printf("ServerRoot conflicts with UserWebsites.\n");
				break;
			case mc_ID_CONFLICT:
				printf("ServerId conflicts with HostId.\n");
				break;
			case mc_NO_HOSTNAME:
				printf("Hostname missing.\n");
				break;
			case mc_NO_PORT:
				printf("BindHTTP "
#ifdef HAVE_SSL
						"or BindHTTPS "
#endif
						"must be specified.\n");
				break;
			case mc_REQUIRE:
				printf("RequireSSL and RequiredBinding not allowed outside VirutalHost block.\n");
				break;
			default:
				printf("invalid option in line %d.\n", code);
		}   
	}
}

bool replace_variables(char **line) {
	bool replaced = false;
	t_keyvalue *variable;
	char *new;

	variable = variables;
	while (variable != NULL) {
		if ((new = str_replace(*line, variable->key, variable->value)) != NULL) {
			if (replaced) {
				free(*line);
			}
			*line = new;
			replaced = true;
		}
		variable = variable->next;
	}

	return replaced;
}

int read_main_configfile(char *file, t_config *config) {
	int  retval = 0, counter = 0;
	FILE *fp;
	char *line, *key, *value;
	enum t_section {none, virtualhost, directory} section = none;
	bool variables_replaced;
	t_host *current_host;
	t_directory *current_directory = NULL;

	/* Read and parse Hiawatha configurationfile.
	 */
	if ((fp = fopen(file, "r")) != NULL) {
		current_host = config->first_host;
		line = (char*)malloc(257);
		*(line + 256) = '\0';
		while (fgets(line, 256, fp) != NULL) {
			counter++;
			key = strip_line(line);
			if (*key != '\0') {
				variables_replaced = false;
				if (strncmp(key, "set ", 4) == 0) {
					if ((variables = parse_keyvaluelist(key + 4, variables, '=')) == NULL) {
						retval = counter;
					}
				} else if (split_string(key, &key, &value, '=') != -1) {
					if (strlen(value) > 0) {
						key = strlower(key);
						variables_replaced = replace_variables(&value);
						do {
							if (section == none) {
								if (was_system_setting(key, value, config)) {
									break;
								}
							}
							if ((section == none) || (section == virtualhost)) {
								if (was_host_setting(key, value, current_host)) {
									break;
								} else if (was_config_setting(key, value, current_host)) {
									break;
								}
							}
							if (section == directory) {
								if (was_directory_setting(key, value, current_directory)) {
									break;
								}
							}
							retval = counter;
						} while (false);
					} else {
						retval = counter;
					}
				} else {
					key = strlower(key);
					if (strcmp(key, "<virtualhost>") == 0) {
						if (section == none) {
							while (current_host->next != NULL) {
								current_host = current_host->next;
							}
							current_host->next = new_host();
							current_host = current_host->next;
							section = virtualhost;
						} else {
							retval = counter;
						}
					} else if (strcmp(key, "</virtualhost>") == 0) {
						if (section == virtualhost) {
							if (current_host->Hostname.size > 0) {
								current_host = config->first_host;
								section = none;
							} else {
								retval = counter;
							}
						} else {
							retval = counter;
						}
					} else if (strcmp(key, "<directory>") == 0) {
						if (section == none) {
							if (config->Directory != NULL) {
								current_directory = config->Directory;
								while (current_directory->next != NULL) {
									current_directory = current_directory->next;
								}
								current_directory->next = new_directory();
								current_directory = current_directory->next;
							} else {
								config->Directory = new_directory();
								current_directory = config->Directory;
							}
							section = directory;
						} else {
							retval = counter;
						}
					} else if (strcmp(key, "</directory>") == 0) {
						if (section == directory) {
							if (config->Directory->Path != NULL) {
								current_directory = NULL;
								section = none;
							} else {
								retval = counter;
							}
						} else {
							retval = counter;
						}
					} else if (strncmp(key, "include ", 8) == 0) {
						value = key + 8;
						variables_replaced = replace_variables(&value);
						if (section == none) {
							if (including == false) {
								including = true;
								if ((retval = read_main_configfile(value, config)) != 0) {
									print_configerror(value, retval);
									retval = mc_INCLUDE;
								}
								including = false;
							} else {
								retval = counter;
							}
						} else {
							retval = counter;
						}
					} else {
						retval = counter;
					}
				}
				if (variables_replaced) {
					free(value);
				}
			}
			if (retval != 0) {
				break;
			}
		}
		fclose(fp);
		free(line);
		if (including == false) {
			remove_keyvaluelist(variables);
			variables = NULL;
		}

		if ((retval == 0) && (section != none)) {
			retval = counter;
		}
	} else {
		retval = mc_READ_ERROR;
	}

	return retval;
}

t_config *remove_config(t_config *config) {
	t_host *host;
	t_directory *dir;
	t_CGIhandler *CGI;
	t_denybotlist *bot;

	if (config != NULL) {
		sfree(config->ServerRoot);
		remove_accesslist(config->LogfileMask);
		remove_accesslist(config->BanlistMask);
		remove_charlist(&(config->CGIextension));
#ifdef HAVE_PLUGIN
		sfree(config->PluginURI);
#endif

		while (config->first_host != NULL) {
			host = config->first_host;
			config->first_host = config->first_host->next;

			sfree(host->ErrorHandler);
			remove_charlist(&(host->Hostname));
			sfree(host->PasswordFile);
			sfree(host->Groups.array);
			remove_charlist(&(host->Volatile));
			remove_charlist(&(host->ImageReferer));
			sfree(host->ImgRefReplacement);
			remove_charlist(&(host->RequiredBinding));
			remove_accesslist(host->AccessList);
			remove_keyvaluelist(host->EnvirStr);
			sfree(host->GroupFile);
			remove_charlist(&(host->RequiredGroup));
			while (host->DenyBot != NULL) {
				bot = host->DenyBot;
				host->DenyBot = host->DenyBot->next;

				sfree(bot->bot);
				remove_charlist(&(bot->uri));
				free(bot);
			}
			free(host);
		}
		while (config->Directory != NULL) {
			dir = config->Directory;
			config->Directory = config->Directory->next;

			sfree(dir->Path);
			sfree(dir->Groups.array);
			sfree(dir->PasswordFile);
			sfree(dir->GroupFile);
			remove_charlist(&(dir->RequiredGroup));
			remove_charlist(&(dir->ImageReferer));
			sfree(dir->ImgRefReplacement);
			remove_keyvaluelist(dir->EnvirStr);
			free(dir);
		}
		while (config->CGIhandler != NULL) {
			CGI = config->CGIhandler;
			config->CGIhandler = config->CGIhandler->next;

			free(CGI->handler);
			remove_charlist(&(CGI->extension));
			free(CGI);
		}
		free(config);
	}

	return NULL;
}

t_homedir *read_homedirs(char *configfile) {
	t_homedir *homedir = NULL, *new;
	FILE *fp;
	char *line, *field, *rest;

	if ((fp = fopen(configfile, "r")) != NULL) {
		if ((line = (char*)malloc(257)) != NULL) {
			*(line + 256) = '\0';
			while (fgets(line, 256, fp) != NULL) {
				rest = strip_line(line);
				if (*rest != '\0') {
					if ((new = (t_homedir*)malloc(sizeof(t_homedir))) == NULL) {
						break;
					} else if (split_string(rest, &field, &rest, ':') == -1) {
						free(new);
					} else if ((new->Username = strdup(field)) == NULL) {
						free(new);
						break;
					} else if (split_string(rest, &field, &rest, ':') == -1) {
						free(new->Username);
						free(new);
					} else if ((new->UserId = (uid_t)str2int(field)) <= 0) {
						free(new->Username);
						free(new);
					} else if (split_string(rest, &field, &rest, ':') == -1) {
						free(new->Username);
						free(new);
					} else if ((new->GroupId = (gid_t)str2int(field)) <= 0) {
						free(new->Username);
						free(new);
					} else {
						split_string(rest, &field, &rest, ':');
						if ((*field != '/') || (strlen(field) > 128) || (*(field + strlen(field) - 1) == '/')) {
							free(new->Username);
							free(new);
						} else if ((new->Directory = strdup(field)) == NULL) {
							free(new->Username);
							free(new);
							break;
						} else {
							new->next = homedir;
							homedir = new;
						}
					}
				}
			}
			free(line);
		}
		fclose(fp);
	}

	return homedir;
}

t_homedir *remove_homedirs(t_homedir *homedir) {
	t_homedir *old;

	while (homedir != NULL) {
		old = homedir;
		homedir = homedir->next;

		sfree(old->Username);
		sfree(old->Directory);
		free(old);
	}

	return NULL;
}

int read_user_configfile(char *file, t_host *host) {
	int  retval = 0;
	FILE *fp;
	char *line, *key, *value;

	if ((fp = fopen(file, "r")) != NULL) {
		if ((line = (char*)malloc(257)) != NULL) {
			*(line + 256) = '\0';
			while (fgets(line, 256, fp) != NULL) {
				key = strip_line(line);
				if (*key != '\0') {
					if (split_string(key, &key, &value, '=') != -1) {
						key = strlower(key);
						was_config_setting(key, value, host);
					}
				}
			}
			free(line);
		} else {
			retval = -1;
		}
		fclose(fp);
	}

	return retval;
}

#ifdef HAVE_SSL
t_host *get_hostrecord(t_host *host, char *hostname, char *bindname, bool via_ssl) {
#else
t_host *get_hostrecord(t_host *host, char *hostname, char *bindname) {
#endif
	char *dots;
	int i, len1, len2;

	if (hostname != NULL) {
		if ((dots = strchr(hostname, ':')) != NULL) {
			*dots = '\0';
		}

		len1 = strlen(hostname);
		while (host != NULL) {
			for (i = 0; i < host->Hostname.size; i++) {
#ifdef HAVE_SSL
				if (host->RequireSSL && (via_ssl == false)) {
					// Skip host
					continue;
				} else
#endif
				if (strcmp(hostname, *(host->Hostname.item + i)) == 0) {
					// Exact match
					return host;
				} else if (strncmp(*(host->Hostname.item + i), "*.", 2) == 0) {
					// Wildcard in configuration
					if (strcmp(hostname, *(host->Hostname.item + i) + 2) == 0) {
						// Only domainname requested
						return host;
					} else {
						len2 = strlen(*(host->Hostname.item + i));
						if (len1 >= len2) {
							if (strcmp(hostname + len1 - len2 + 1, *(host->Hostname.item + i) + 1) == 0) {
								// Wildcard match for hostname
								return host;
							}
						}
					}
				}
			}

			host = host->next;
		}
	}

	return NULL;
}

t_throttle *read_throttleconfig(char *configfile) {
	FILE *fp;
	char *line, *key, *value;
	t_throttle *throttle = NULL, *throt = NULL;
	int speed;

	if ((fp = fopen(configfile, "r")) != NULL) {
		line = (char*)malloc(257);
		*(line + 256) = '\0';
		while (fgets(line, 256, fp) != NULL) {
			if ((*line != '#') && (strlen(line) > 0)) {
				if (split_string(line, &key, &value, ':') != -1) {
					if (((*key == '.') || (strchr(key, '/') != NULL)) && (speed = str2int(value)) > 0) {
						if (throt == NULL) {
							throttle = throt = (t_throttle*)malloc(sizeof(t_throttle));
						} else {
							throt->next = (t_throttle*)malloc(sizeof(t_throttle));
							throt = throt->next;
						}
						throt->next = NULL;
						throt->filetype = strlower(strdup(key));
						throt->upload_speed = speed << 10; // convert to kilobytes
					}
				}
			}
		}
		fclose(fp);
		free(line);

		return throttle;
	} else {
		return NULL;
	}
}

unsigned short get_throttlespeed(char *type, t_throttle *throttle) {
	t_throttle *throt;
	unsigned long speed = 0;
	int len_type, len_throt;
	char *type_lower;

	if (type != NULL) {
		type_lower = strlower(strdup(type));
		len_type = strlen(type);

		throt = throttle;
		while (throt != NULL) {
			len_throt = strlen(throt->filetype);
			if (len_type >= len_throt) {
				if (memcmp(throt->filetype, type_lower, len_throt) == 0) {
					speed = throt->upload_speed;
					break;
				}
			}
			throt = throt->next;
		}
		free(type_lower);
	}

	return speed;
}

t_throttle *remove_throttleconfig(t_throttle *throttle) {
	t_throttle *old;

	while (throttle != NULL) {
		old = throttle;
		throttle = throttle->next;

		sfree(old->filetype);
		free(old);
	}

	return NULL;
}
