#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <pwd.h>
#include "libstr.h"
#include "session.h"
#include "liblist.h"
#ifdef HAVE_SSL
#include "libssl.h"
#endif

int new_child_id = 0;

#ifdef HAVE_COMMAND
extern unsigned long clients_served;
#endif

/* Set the entries in a session-record to the default values.
 */
static void clear_session(t_session *session) {
	session->HEAD_request = false;
	session->is_CGI_script = false;
	session->CGI_handler = NULL;
	session->method = NULL;
	session->URI = NULL;
	session->URI_is_dir = false;
	session->accept_gzip = false;
	session->encode_gzip = false;
	session->vars = NULL;
	session->vars_copied = false;
	session->http_version = NULL;
	session->headerfields = NULL;
	session->body = NULL;
	session->body_copied = false;
	session->header_length = 0;
	session->content_length = 0;
	session->file_on_disk = NULL;
	session->mimetype = NULL;
	session->host = session->config->first_host;
	session->host_copied = false;
	session->throttle = 0;
	session->throttle_timer = 0;
	session->bytecounter = 0;
	session->part_of_dirspeed = false;
    session->chunkbuffer = NULL;
	session->chunksize = 0;
	session->remote_user = NULL;
	session->Directory = NULL;
	session->handling_error = false;
	session->cookie = NULL;
	session->cookie_copied = false;
	session->websiteroot_copied = false;
}

/* Initialize a session-record.
 */
void init_session(t_session *session) {
#ifdef HAVE_COMMAND
	clients_served++;
#endif
	if ((session->child_id = new_child_id++) == MAX_CHILD_ID) {
		new_child_id = 0;
	}
	session->force_quit = false;
	session->kept_alive = 0;
	session->flooding_counter = 0;
	session->flooding_timer = session->config->FloodingTime;
#ifdef HAVE_SSL
	session->ssl_data = NULL;
#endif

	session->request = NULL;
	session->buffer_size = 0;
	session->bytes_in_buffer = 0;
	session->connection_id[0] = '\0';

	clear_session(session);
}

/* Reset a session-record for reuse.
 */
void reset_session(t_session *session) {
	int size;

	sfree(session->remote_user);
	sfree(session->file_on_disk);
	sfree(session->chunkbuffer);
	remove_headerfields(session);
	if (session->Directory != NULL) {
		pthread_mutex_lock(&(session->Directory->client_mutex));
		if (session->part_of_dirspeed) {
			if (--session->Directory->Clients == 0) {
				session->Directory->SessionSpeed = session->Directory->UploadSpeed;
			} else {
				session->Directory->SessionSpeed = session->Directory->UploadSpeed / session->Directory->Clients;
			}
		}
		pthread_mutex_unlock(&(session->Directory->client_mutex));
	}
	if (session->vars_copied) {
		free(session->vars);
	}
	if (session->body_copied) {
		free(session->body);
	}
	if (session->cookie_copied) {
		free(session->cookie);
	}

	/* HTTP pipelining
	 */
	size = session->header_length + session->content_length;
	if ((session->bytes_in_buffer > size) && session->keep_alive) {
		session->bytes_in_buffer -= size;
		memcpy(session->request, session->request + size, session->bytes_in_buffer);
		*(session->request + session->bytes_in_buffer) = '\0';
	} else {
		sfree(session->request);
		session->request = NULL;
		session->buffer_size = 0;
		session->bytes_in_buffer = 0;
	}
	
	clear_session(session);
	session->flooding_counter++;
}

/* Parse the HTTP headerfields from the received request.
 */
t_headerfield *parse_headerfields(char *line) {
	t_headerfield *first, *headerfield;
	char *value;

	if ((first = headerfield = (t_headerfield*)malloc(sizeof(t_headerfield))) == NULL) {
		return NULL;
	}
	headerfield->data = line;
	headerfield->next = NULL;
	while (*line != '\0') {
		if (*line == '\r') {
			*line = '\0';
			if ((*(line + 1) == '\n') && (*(line + 2) != '\r') && (*(line + 2) != '\0')) {
				if ((headerfield->next = (t_headerfield*)malloc(sizeof(t_headerfield))) == NULL) {
					return first;
				}
				headerfield = headerfield->next;
				headerfield->next = NULL;
				headerfield->data = line + 2;
			}
		}
		line++;
	}

	headerfield = first;
	while (headerfield != NULL) {
		if ((value = strchr(headerfield->data, ':')) != NULL) {
			do {
				value++;
			} while ((*value == ' ') && (*value != '\0'));
			headerfield->value_offset = (value - headerfield->data);
		} else {
			headerfield->value_offset = 0;
		}
		headerfield = headerfield->next;
	}

	return first;
}

/* Search for a headerfield and return its value.
 */
char *get_headerfield(char *key, t_headerfield *headerfields) {
	char *retval = NULL;
	int len;

	len = strlen(key);
	while (headerfields != NULL) {
		if (str_ncompare(headerfields->data, key, len) == 0) {
			retval = headerfields->data + headerfields->value_offset;
			break;
		}
		headerfields = headerfields->next;
	}

	return retval;
}

/* free() a list of headerfields.
 */
void remove_headerfields(t_session *session) {
	t_headerfield *to_be_removed;

	while (session->headerfields != NULL) {
		to_be_removed = session->headerfields;
		session->headerfields = session->headerfields->next;

		free(to_be_removed);
	}
}

/* Return the path of the user's homedirectory.
 */
int get_homedir(t_session *session, char *username) {
	t_homedir *homedir;
	struct passwd *pwd;

	if (session->config->HomedirSource == hiaconf) {
		homedir = session->config->Homedir;
		while (homedir != NULL) {
			if (strcmp(username, homedir->Username) == 0) {
				session->host->UserId = homedir->UserId;
				session->host->GroupId = homedir->GroupId;
				session->host->WebsiteRoot = homedir->Directory;
				return 200;
			}
			homedir = homedir->next;
		}
	} else if ((pwd = getpwnam(username)) != NULL) {
		session->host->UserId = pwd->pw_uid;
		session->host->GroupId = pwd->pw_gid;
		session->host->WebsiteRoot = (char*)malloc(strlen(pwd->pw_dir) + 13);
		strcpy(session->host->WebsiteRoot, pwd->pw_dir);
		strcat(session->host->WebsiteRoot, "/public_html");
		session->websiteroot_copied = true;
		return 200;
	}

	return 404;
}

/* Dupliacte the active host-record. The duplicate can now savely be altered
 * and will be used during the session.
 */
bool duplicate_host(t_session *session) {
	t_host *new_host;

	if ((session->host != NULL) && (session->host_copied == false)) {
		if ((new_host = (t_host*)malloc(sizeof(t_host))) != NULL) {
			memcpy(new_host, session->host, sizeof(t_host));
			new_host->next = NULL;
			session->host = new_host;
			session->host_copied = true;
		} else {
			return false;
		}
	}

	return true;
}

/* Restore the active host-record.
 */
void restore_host(t_session *session) {
	if (session->host_copied) {
		if (session->websiteroot_copied) {
			free(session->host->WebsiteRoot);
			session->websiteroot_copied = false;
		}

		free(session->host);
		session->host_copied = false;
	}
	session->host = session->config->first_host;
}

/* Is the requested file marked as volatile?
 */
bool is_volatile_object(t_session *session) {
	bool retval = false;
	int i;

	for (i = 0; i < session->host->Volatile.size; i++) {
		if (strcmp(session->file_on_disk, *(session->host->Volatile.item + i)) == 0) {
			retval = true;
			break;
		}
	}

	return retval;
}

/* Copy the settings from a directory-record to the active host-record.
 */
int copy_directory_settings(t_session *session) {
	int path_length;
	t_directory *dir;
	bool match;

	dir = session->config->Directory;
	while (dir != NULL) {
		path_length = strlen(dir->Path);
		if (strlen(session->file_on_disk) >= path_length) {
			if (dir->PathMatch == root) {
				match = (strncmp(session->file_on_disk, dir->Path, path_length) == 0);
			} else {
				match = (strstr(session->file_on_disk, dir->Path) != NULL);
			}
			if (match) {
				if (duplicate_host(session)) {
					session->Directory = dir;

					if (dir->MaxClients > -1) {
						pthread_mutex_lock(&(dir->client_mutex));
						if (dir->Clients < dir->MaxClients) {
							session->throttle = dir->SessionSpeed = dir->UploadSpeed / ++dir->Clients;
							pthread_mutex_unlock(&(dir->client_mutex));
							session->part_of_dirspeed = true;
						} else {
							pthread_mutex_unlock(&(dir->client_mutex));
							return 503;
						}
					}

					if (dir->UserId != (uid_t)-1) {
						session->host->UserId = dir->UserId;
						session->host->GroupId = dir->GroupId;
						session->host->Groups.number = dir->Groups.number;
						session->host->Groups.array = dir->Groups.array;
					}
					if (dir->ExeCGIset) {
						session->host->ExeCGI = dir->ExeCGI;
					}
					if (dir->ShowIndexSet) {
						session->host->ShowIndex = dir->ShowIndex;
					}
					if (dir->FollowSymlinksSet) {
						session->host->FollowSymlinks = dir->FollowSymlinks;
					}
					if (dir->AccessList != NULL) {
						session->host->AccessList = dir->AccessList;
					}
					if (dir->ImageReferer.size > 0) {
						session->host->ImageReferer.size = dir->ImageReferer.size;
						session->host->ImageReferer.item = dir->ImageReferer.item;
						session->host->ImgRefReplacement = dir->ImgRefReplacement;
					}
					if (dir->PasswordFile != NULL) {
						session->host->PasswordFile = dir->PasswordFile;
					}
					if (dir->GroupFile != NULL) {
						session->host->GroupFile = dir->GroupFile;
						session->host->RequiredGroup = dir->RequiredGroup;
					}
					break;
				} else {
					return 500;
				}
			}
		}
		dir = dir->next;
	}

	return 200;
}

/* Copy a headerfield to an environment setting
 */
static void headerfield_to_environment(t_session *session, char *key, char *envir) {
	char *value;

	if ((value = get_headerfield(key, session->headerfields)) != NULL) {
		setenv(envir, value, 1);
	}
}

/* Set environment variables for CGI script.
 */
void set_environment(t_session *session) {
	char ip[16], len[16], port[16];
	unsigned char *ip_address;
	t_keyvalue *envir;

	setenv("GATEWAY_INTERFACE", "CGI/1.1", 1);
	setenv("DOCUMENT_ROOT", session->host->WebsiteRoot, 1);
	setenv("REQUEST_METHOD", session->method, 1);
	setenv("REQUEST_URI", session->URI, 1);
	setenv("SCRIPT_NAME", session->URI, 1);
	setenv("SCRIPT_FILENAME", session->file_on_disk, 1);

	ip_address = (unsigned char*)&(session->ip_address);
	sprintf(ip, "%hhu.%hhu.%hhu.%hhu", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
	setenv("REMOTE_ADDR", ip, 1);

	if (session->vars != NULL) {
		setenv("QUERY_STRING", session->vars, 1);
	}

	if (session->body != NULL) {
		sprintf(len, "%ld", session->content_length);
		setenv("CONTENT_LENGTH", len, 1);
		headerfield_to_environment(session, "Content-Type:", "CONTENT_TYPE");
	}

	sprintf(port, "%d", session->binding->port);
	setenv("SERVER_NAME", *(session->host->Hostname.item), 1);
	setenv("SERVER_PORT", port, 1);
	setenv("SERVER_PROTOCOL", session->http_version, 1);
	setenv("SERVER_SOFTWARE", "Hiawatha v"VERSION, 1);
	setenv("REDIRECT_STATUS", "CGI", 1);

	if (session->remote_user != NULL) {
		setenv("AUTH_TYPE", "Basic", 1);
		setenv("REMOTE_USER", session->remote_user, 1);
	}

#ifdef HAVE_SSL
	if (session->via_ssl) {
		setenv("CONNECTION_TYPE", "SSL", 1);
	} else
#endif
		setenv("CONNECTION_TYPE", "plain", 1);

	if (session->connection_id[0] != '\0') {
		setenv("CONNECTION_ID", session->connection_id, 1);
	}

	headerfield_to_environment(session, "Accept:", "HTTP_ACCEPT");
	headerfield_to_environment(session, "Accept-Charset:", "HTTP_ACCEPT_CHARSET");
	headerfield_to_environment(session, "Accept-Language:", "HTTP_ACCEPT_LANGUAGE");
	headerfield_to_environment(session, "Client-IP:", "HTTP_CLIENT_IP");
	headerfield_to_environment(session, "From:", "HTTP_FROM");
	headerfield_to_environment(session, "Host:", "HTTP_HOST");
	headerfield_to_environment(session, "Range:", "HTTP_RANGE");
	headerfield_to_environment(session, "Referer:", "HTTP_REFERER");
	headerfield_to_environment(session, "User-Agent:", "HTTP_USER_AGENT");
	headerfield_to_environment(session, "Via:", "HTTP_VIA");
	headerfield_to_environment(session, "X-Forwarded-For:", "HTTP_X_FORWARDED_FOR");

	if (session->cookie != NULL) {
		setenv("HTTP_COOKIE", session->cookie, 1);
	} else {
		headerfield_to_environment(session, "Cookie:", "HTTP_COOKIE");
	}

	envir = session->host->EnvirStr;
	while (envir != NULL) {
		setenv(envir->key, envir->value, 1);
		envir = envir->next;
	}
}

/* Complete path in chrooted session?
 *  *  */
char *file_in_chroot(t_session *session, char *file) {
	int length;

	if (session->config->ServerRoot != NULL) {
		length = strlen(session->config->ServerRoot);
		if (strlen(session->host->PasswordFile) > length) {
			if (memcmp(session->host->PasswordFile, session->config->ServerRoot, length) == 0) {
				file += length;
			}
		}
	}

	return file;
}

/* Check if User-Agent string contains DenyBot substring.
 */
bool client_is_rejected_bot(t_session *session) {
	int i, urilen, len;
	char *useragent;
	t_denybotlist *botlist;

	if (session->host->DenyBot == NULL) {
		return false;
	} else if ((useragent = get_headerfield("User-Agent:", session->headerfields)) == NULL) {
		return false;
	}

	urilen = strlen(session->URI);
	botlist = session->host->DenyBot;
	while (botlist != NULL) {
		if (strstr(useragent, botlist->bot) != NULL) {
			for (i = 0; i < botlist->uri.size; i++) {
				len = strlen(*(botlist->uri.item + i));
				if (urilen >= len) {
					if (memcmp(*(botlist->uri.item + i), session->URI, len) == 0) {
						return true;
					}
				}
			}
		}
		botlist = botlist->next;
	}

	return false;
}

void close_socket(t_session *session) {
	if (session->socket_open) {
#ifdef HAVE_SSL
		if (session->via_ssl) {
			sslClose(session->ssl_data);
		}
#endif
		close(session->client_socket);
		session->socket_open = false;
	}
}
