/* Hiawatha | target.c
 *
 * All the routines for sending the requested item back to the client.
 */
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <grp.h>
#include <time.h>
#include <signal.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#include "target.h"
#include "log.h"
#include "libstr.h"
#include "send.h"
#ifdef HAVE_PLUGIN
#include "plugin.h"
#endif

#define MAX_VOLATILE_SIZE    1048576
#define FILE_BUFFER_SIZE       32768
#define CGI_BUFFER_SIZE         1024
#define MAX_CGI_HEADER          8192
#define MAX_TRACE_HEADER        1024

#define rs_QUIT       -1
#define rs_DISCONNECT -2
#define rs_FORCE_QUIT -3

char *chunked      = "Transfer-Encoding: chunked\r\n";
char *hs_allow     = "Allow: GET, HEAD, OPTIONS";
char *hs_post      = ", POST";

typedef struct type_filelist {
	char   *name;
	int    size;
	time_t time;
	bool   is_dir;

	struct type_filelist *next;
} t_filelist;

extern char *hs_eol;
extern char *hs_conlen;
extern char *hs_contyp;

#ifdef HAVE_COMMAND
extern unsigned long req_files, req_cgi, req_index;
#endif

/* Read a file from disk and send it to the client.
 */
int send_file(t_session *session) {
	char *referer, *buffer = NULL, value[64], *dot, *date, *range, *range_start, *range_end;
	long bytes_read, total_bytes, size, file_size, send_start, send_end, send_size, speed;
	int  retval, handle, http_code;
	bool invalid_referer = false;
	struct stat status;
	struct tm *fdate;

#ifdef HAVE_COMMAND
	req_files++;
#endif
	http_code = 200;

	session->mimetype = get_mimetype(session->file_on_disk, session->config->Mimetype);

	if (session->host->ImageReferer.size > 0) {
		if (memcmp(session->mimetype, "image/", 6) == 0) {
			if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
				return 404;
			} else {
				close(handle);
			}

			invalid_referer = true;
			if ((referer = get_headerfield("Referer:", session->headerfields)) != NULL) {
				if (memcmp(referer, "http://", 7) == 0) {
					referer += 7;
					if ((dot = strchr(referer, '/')) != NULL) {
						*dot = '\0';
					}
					for (size = 0; size < session->host->ImageReferer.size; size++) {
						if (strstr(referer, *(session->host->ImageReferer.item + size)) != NULL) {
							invalid_referer = false;
							break;
						}
					}
					if (dot != NULL) {
						*dot = '/';
					}
				}
			}
		}
	}
	if (invalid_referer) {
		free(session->file_on_disk);
		if ((session->file_on_disk = (char*)malloc(strlen(session->host->ImgRefReplacement) + 4)) != NULL) { // + 3 for ".gz" (gzip encoding)
			strcpy(session->file_on_disk, session->host->ImgRefReplacement);
			session->mimetype = get_mimetype(session->file_on_disk, session->config->Mimetype);
		} else {
			return 500;
		}
	}
	
	handle = -1;
	if (session->accept_gzip) {
		size = strlen(session->file_on_disk);
		strcpy(session->file_on_disk + size, ".gz");
		if ((handle = open(session->file_on_disk, O_RDONLY)) != -1) {
			session->encode_gzip = true;
		} else {
			*(session->file_on_disk + size) = '\0';
		}
	}
	if (handle == -1) {
		if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
			if (session->config->ServerUid != (uid_t)-1) {
				if (errno == EACCES) {
					return 403;
				}
			}
			return 404;
		}
	}

	do {
		if (session->config->ServerUid == (uid_t)-1) {
			switch (fs_access_allowed(session->file_on_disk, session->host->UserId, session->host->GroupId, &(session->host->Groups), file)) {
				case error:
					retval = 500;
					break;
				case no:
					retval = 403;
					break;
				case yes:
					break;
			}
		}

		if (session->host->FollowSymlinks == false) {
			switch (contains_symlink(session->file_on_disk)) {
				case error:
					retval = 500;
					break;
				case no:
					break;
				case yes:
					retval = 403;
					break;
			}
		}

		/* Modified-Since
		 */
		if (session->handling_error == false) {
			if ((date = get_headerfield("If-Modified-Since:", session->headerfields)) != NULL) {
				if (if_modified_since(handle, date) == 0) {
					retval = 304;
					break;
				}
			} else if ((date = get_headerfield("If-Unmodified-Since:", session->headerfields)) != NULL) {
				if (if_modified_since(handle, date) == 1) {
					retval = 412;
					break;
				}
			}
		}

		/* Set throttlespeed
		 */
		dot = session->URI + strlen(session->URI);
		while ((*dot != '.') && (dot != session->URI)) {
			dot--;
		}
		if (*dot == '.') {
			if ((speed = get_throttlespeed(dot, session->config->Throttle)) != 0) {
				if ((session->throttle == 0) || (speed < session->throttle)) {
					session->throttle = speed;
				}
			}
			if ((speed = get_throttlespeed(session->mimetype, session->config->Throttle)) != 0) {
				if ((session->throttle == 0) || (speed < session->throttle)) {
					session->throttle = speed;
				}
			}
		}

		if ((file_size = filesize(session->file_on_disk)) == -1) {
			retval = 500;
			break;
		}
		send_start = 0;
		send_end = file_size - 1;
		send_size = file_size;

		/* Range
		 */
		if ((range = get_headerfield("Range:", session->headerfields)) != NULL) {
			if (memcmp(range, "bytes=", 6) == 0) {
				range += 6;
				if (split_string(range, &range_start, &range_end, '-') == 0) {

					if (*range_start != '\0') {
						if ((send_start = str2int(range_start)) >= 0) {
							if (*range_end != '\0') {
								if ((send_end = str2int(range_end)) >= 0) {
									// bytes=XX-XX
									http_code = 206;
								}
							} else {
								// bytes=XX-
								http_code = 206;
							}
						}
					} else {
						if ((send_start = str2int(range_end)) >= 0) {
							// bytes=-XX
							send_start = file_size - send_start;
							http_code = 206;
						}
					}

					if (http_code == 206) {
						if (send_start >= file_size) {
							retval = 416;
							break;
						}
						if (send_start < 0) {
							send_start = 0;
						}
						if (send_end >= file_size) {
							send_end = file_size - 1;
						}
						if (send_start <= send_end) {
							send_size = send_end - send_start + 1;
						} else {
							retval = 416;
							break;
						}
					} else {
						send_start = 0;
						send_end = file_size - 1;
						send_size = file_size;
					}
				}
			}
		}

		retval = -1;
		if (send_header(session, http_code) == -1) {
			break;
		}

		/* Last-Modified
		 */
		if (fstat(handle, &status) == -1) {
			break;
		} else if ((fdate = gmtime(&(status.st_mtime))) == NULL) {
			break;
		} else if (send_buffer(session, "Last-Modified: ", 15) == -1) {
			break;
		} else if (strftime(value, 64, "%a, %d %b %Y %X GMT\r\n", fdate) == 0) {
			break;
		} else if (send_buffer(session, value, strlen(value)) == -1) {
			break;
		}

		/* Content-Range
		 */
		if (http_code == 206) {
			if (send_buffer(session, "Content-Range: bytes ", 21) == -1) {
				break;
			} else if (sprintf(value, "%ld-%ld/%ld\r\n", send_start, send_end, file_size) == -1) {
				break;
			} else if (send_buffer(session, value, strlen(value)) == -1) {
				break;
			}
		}

		if (send_buffer(session, hs_conlen, 16) == -1) {
			break;
		} else if (sprintf(value, "%ld\r\n\r\n", send_size) == -1) {
			break;
		} else if (send_buffer(session, value, strlen(value)) == -1) {
			break;
		}

		retval = 200;
		if (session->HEAD_request == false) {
			if (send_start > 0) {
				lseek(handle, send_start, SEEK_SET);
			}
			if (is_volatile_object(session) && (file_size <= MAX_VOLATILE_SIZE)) {
				/* Volatile object
				 */
				if ((buffer = (char*)malloc(send_size)) != NULL) {
					total_bytes = 0;
					do {
						if ((bytes_read = read(handle, buffer + total_bytes, send_size - total_bytes)) == -1) {
							if (errno == EINTR) {
								bytes_read = 0;
							}
						} else {
							total_bytes += bytes_read;
						}
					} while ((bytes_read != -1) && (total_bytes < send_size));
					if (bytes_read != -1) {
						if (send_buffer(session, buffer, send_size) == -1) {
							retval = -1;
						}
					} else {
						retval = -1;
					}
				} else {
					retval = -1;
				}
			} else {
				/* Normal file
				 */
#ifdef HAVE_SENDFILE
				off_t offset;

				if (session->throttle == 0) {
					offset = send_start;
					sendfile(session->client_socket, handle, &offset, send_size);
				} else 
#endif
				if ((buffer = (char*)malloc(FILE_BUFFER_SIZE)) != NULL) {
					while ((send_size > 0) && (retval == 200)) {
						switch ((bytes_read = read(handle, buffer, FILE_BUFFER_SIZE))) {
							case -1:
								if (errno != EINTR) {
									retval = -1;
								}
								break;
							case 0:
								send_size = 0;
								break;
							default:
								if (bytes_read > send_size) {
									bytes_read = send_size;
								}
								if (send_buffer(session, buffer, bytes_read) == -1) {
									retval = -1;
								}
								send_size -= bytes_read;
						}
					}
				} else {
					retval = -1;
				}
			}
			if (buffer != NULL) {
				free(buffer);
			}
		}
	} while (false);
	close(handle);

	return retval;
}

/* Run a CGI program and send output to the client.
 */
int run_script(t_session *session, int code) {
	int retval = 200, result, handle, timer, CGI_pid, to_cgi, from_cgi;
	int post_pipe[2], html_pipe[2], error_pipe[2], cgi_error, highest_fd;
	long buffer_size = CGI_BUFFER_SIZE + 1, total_bytes = 0, bytes_read = 0, error_bytes;
	char *line, *error_line = NULL, *new_line, *dir, *c, *contyp_start, *contyp_end, errorcode[4];
	bool in_body = false, send_in_chunks = true;
 	struct timeval select_timeout;
	enum t_extbool access;
	fd_set read_fds;
#ifdef DEBUG
	unsigned char random[CID_LENGTH];
#endif

#ifdef HAVE_COMMAND
	req_cgi++;
#endif

	/* Proxies that speak HTTP/1.0 and don't understand chunked Transfer-Encoding.
	 */
	if (*(session->http_version + 7) == '0') {
		session->keep_alive = false;
	}

	if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
		if (session->config->ServerUid != (uid_t)-1) {
			if (errno == EACCES) {
				return 403;
			}
		}
		return 404;
	} else {
		close(handle);
	}

	if (session->host->ExeCGI == false) {
		return 403;
	}

	if (session->CGI_handler != NULL) {
		if (session->config->ServerUid == (uid_t)-1) {
			access = fs_access_allowed(session->file_on_disk, session->host->UserId, session->host->GroupId, &(session->host->Groups), file);
		} else {
			access = yes;
		}
	} else if (session->config->ServerUid == (uid_t)-1) {
		access = fs_access_allowed(session->file_on_disk, session->host->UserId, session->host->GroupId, &(session->host->Groups), executable);
	} else {
		access = fs_access_allowed(session->file_on_disk, session->config->ServerUid, session->config->ServerGid, NULL, executable);
	}
	switch (access) {
		case error:
			return 500;
		case no:
			return 403;
		case yes:
			break;
	}

	if (session->host->FollowSymlinks == false) {
		switch (contains_symlink(session->file_on_disk)) {
			case error:
				return 500;
			case no:
				break;
			case yes:
				return 403;
		}
	}

	do {
		if (pipe(post_pipe) != -1) {
			if (pipe(html_pipe) != -1) {
				if (pipe(error_pipe) != -1) {
					break;
				}
				close(html_pipe[0]);
				close(html_pipe[1]);
			}
			close(post_pipe[0]);
			close(post_pipe[1]);
		}
		return 500;
	} while (false);

	if (session->host->PreventXSS) {
		prevent_XSS(session->vars);
	}

	if (session->host->PreventSQLi) {
		/* Prevent SQL injection
		 */
		if (session->vars != NULL) {
			switch (prevent_SQLi(session->vars, strlen(session->vars) + 1, &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->BanOnSQLi > 0) {
						free(line);
						return rr_SQL_INJECTION;
					}
					session->vars = line;
					session->vars_copied = true;
			}
		}

		if (session->body != NULL) {
			session->content_length = parse_special_chars(session->body, session->content_length);
			switch (result = prevent_SQLi(session->body, session->content_length, &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->BanOnSQLi > 0) {
						free(line);
						return rr_SQL_INJECTION;
					} else {
						session->body = line;
						session->content_length = result;
						session->body_copied = true;
					}
			}
		}

		if ((session->cookie = get_headerfield("Cookie:", session->headerfields)) != NULL) {
			parse_special_chars(session->cookie, strlen(session->cookie));
			switch (prevent_SQLi(session->cookie, strlen(session->cookie), &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->BanOnSQLi > 0) {
						free(line);
						return rr_SQL_INJECTION;
					} else {
						session->cookie = line;
						session->cookie_copied = true;
					}
			}
		}
	}

#ifdef DEBUG
	if (session->connection_id[0] == '\0') {
		if ((handle = open("/dev/random", O_RDONLY)) != -1) {
			read(handle, random, CID_LENGTH);
			close(handle);

			for (result = 0; result < CID_LENGTH; result++) {
				sprintf(&session->connection_id[2 * result], "%02hx", random[result]);
			}

			session->connection_id[2 * CID_LENGTH] == '\0';
		} else {
			return 500;
		}
	}
#endif

	switch (CGI_pid = fork()) {
		case -1:
			retval = 500;
			break;
		case 0:
			/* Child, executes CGI program.
			 */
			if (getuid() == 0) {
				if ((setgroups(session->host->Groups.number, session->host->Groups.array) == -1) ||
					(setgid(session->host->GroupId) == -1) ||
					(setuid(session->host->UserId) == -1)) {
						exit(-1);
				}
			}

			dup2(post_pipe[0], STDIN_FILENO);
			dup2(html_pipe[1], STDOUT_FILENO);
			dup2(error_pipe[1], STDERR_FILENO);

			close(post_pipe[0]);
			close(post_pipe[1]);
			close(html_pipe[0]);
			close(html_pipe[1]);
			close(error_pipe[0]);
			close(error_pipe[1]);

			fcntl(0, F_SETFD, 0);
			fcntl(1, F_SETFD, 0);
			fcntl(2, F_SETFD, 0);

			set_environment(session);
			sprintf(errorcode, "%d", code);
			setenv("HTTP_RETURN_CODE", errorcode, 1);

			if ((dir = strdup(session->file_on_disk)) != NULL) {
				if ((c = strrchr(dir, '/')) != NULL) {
					*c = '\0';
					chdir(dir);
				}
				free(dir);
			}
			if (session->CGI_handler != NULL) {
				result = execlp(session->CGI_handler, session->file_on_disk, session->file_on_disk, NULL);
			} else {
				result = execlp(session->file_on_disk, session->file_on_disk, NULL);
			}
			exit(-1);
			break;
		default:
			/* Parent, reads CGI output
			 */
			close(post_pipe[0]);
			close(error_pipe[1]);
			close(html_pipe[1]);

			to_cgi = post_pipe[1];
			from_cgi = html_pipe[0];
			cgi_error = error_pipe[0];

			/* Send POST data to CGI program.
			 */
			if ((session->body != NULL) && (session->content_length > 0)) {
				write(to_cgi, session->body, session->content_length);
			}

			if ((line = (char*)malloc(buffer_size)) == NULL) {
				retval = 500;
			} else if ((error_line = (char*)malloc(CGI_BUFFER_SIZE + 1)) == NULL) {
				free(line);
				retval = 500;
			}

			if (retval != 200) {
				close(to_cgi);
				close(from_cgi);
				close(cgi_error);
				break;
			}

			if (from_cgi > cgi_error) {
				highest_fd = from_cgi;
			} else {
				highest_fd = cgi_error;
			}
			highest_fd++;

			select_timeout.tv_sec = 1;
			select_timeout.tv_usec = 0;
			timer = session->config->TimeForCGI;

			do {
				FD_ZERO(&read_fds);
				FD_SET(from_cgi, &read_fds);
				FD_SET(cgi_error, &read_fds);
				switch ((result = select(highest_fd, &read_fds, NULL, NULL, &select_timeout))) {
					case -1:
						if (errno != EINTR) {
							if (in_body) {
								retval = rs_DISCONNECT;
							} else {
								retval = 500;
							}
						}
						break;
					case 0:
						if (session->force_quit) {
							retval = rs_FORCE_QUIT;
							break;
						} else if (--timer <= 0) {
							if (in_body) {
								retval = rs_DISCONNECT;
							} else {
								retval = 500;
							}
						} else {
							select_timeout.tv_sec = 1;
							select_timeout.tv_usec = 0;
						}
						break;
					default:
						if (FD_ISSET(from_cgi, &read_fds)) {
							/* FD: from_cgi
							 */
							if (in_body) {
								/* Read body.
								 */
								switch ((bytes_read = read(from_cgi, line, CGI_BUFFER_SIZE))) {
									case -1:
										if (errno != EINTR) {
											retval = rs_DISCONNECT;
										}
										break;
									case 0:
										if (send_in_chunks) {
											if (send_chunk(session, NULL, 0) == -1) {
												retval = rs_DISCONNECT;
											}
										}
										break;
									default:
										if (send_in_chunks) {
											if (send_chunk(session, line, bytes_read) == -1) {
												retval = rs_DISCONNECT;
											}
										} else if (send_buffer(session, line, bytes_read) == -1) {
											retval = rs_DISCONNECT;
										}
								}
							} else {
								/* Expand buffer?
								 */
								if (buffer_size - total_bytes < 100) {
									if (buffer_size < MAX_CGI_HEADER) {
										buffer_size += CGI_BUFFER_SIZE;
										if ((new_line = (char*)realloc(line, buffer_size)) != NULL) {
											line = new_line;
										} else {
											retval = 500;
											break;
										}
									} else {
										retval = 413;
										break;
									}
								}

								/* Read header.
								 */
								switch ((bytes_read = read(from_cgi, line + total_bytes, buffer_size - total_bytes))) {
									case -1:
										if (errno != EINTR) {
											retval = rs_DISCONNECT;
										}
										break;
									case 0:
										retval = 500;
										break;
									default:
										total_bytes += bytes_read;
										*(line + total_bytes) = '\0';

										if ((new_line = strstr(line, "\r\n\r\n")) != NULL) {
											if (session->throttle == 0) {
												if ((contyp_start = str_search(line, hs_contyp)) != NULL) {
													if ((contyp_end = strchr(contyp_start, '\r')) != NULL) {
														contyp_start += 14;
														*contyp_end = '\0';
														session->throttle = get_throttlespeed(contyp_start, session->config->Throttle);
														*contyp_end = '\r';
													}
												}
											}
											if (str_search(line, "Location:")) {
												if (send_header(session, 302) == -1) {
													retval = rs_DISCONNECT;
													break;
												}
											} else if (send_header(session, code) == -1) {
												retval = rs_DISCONNECT;
												break;
											}
											if (code == 401) {
												send_basic_auth(session);
											}
											if ((str_search(line, hs_conlen) != NULL) || (session->keep_alive == false)) {
												send_in_chunks = false;
											} else if (send_buffer(session, chunked, 28) == -1) {
												retval = rs_DISCONNECT;
												break;
											}

											/* Send the header.
											 */
											if (send_in_chunks || session->HEAD_request) {
												new_line += 4;
												result = new_line - line;
												if (send_buffer(session, line, result) == -1) {
													retval = rs_DISCONNECT;
													break;
												}
												if (session->HEAD_request) {
													retval = rs_QUIT;
													break;
												}
												result = total_bytes - result;
												if (send_chunk(session, new_line, result) == -1) {
													retval = rs_DISCONNECT;
													break;
												}
											} else if (send_buffer(session, line, total_bytes) == -1) {
												retval = rs_DISCONNECT;
												break;
											}
											free(line);
											if ((line = (char*)malloc(CGI_BUFFER_SIZE)) != NULL) {
												in_body = true;
											} else {
												retval = rs_DISCONNECT;
												break;
											}
										}
								}
							}
						}

						if (FD_ISSET(cgi_error, &read_fds)) {
							/* FD: cgi_error
							 */
							switch ((error_bytes = read(cgi_error, error_line, CGI_BUFFER_SIZE))) {
								case -1:
									if (errno != EINTR) {
										retval = rs_DISCONNECT;
									}
									break;
								case 0:
									break;
								default:
									*(error_line + error_bytes) = '\0';
									log_CGI_error(session->config->SystemLogfile, session->file_on_disk, error_line);
							}
						}

				} // switch
			} while ((retval == 200) && (bytes_read > 0));

			kill(CGI_pid, 9); // Just to be sure.
			switch (retval) {
				case rs_DISCONNECT:
				case rs_FORCE_QUIT:
					session->keep_alive = false;
				case rs_QUIT:
					retval = 200;
			}

			close(to_cgi);
			close(from_cgi);
			close(cgi_error);
			if (line != NULL) {
				free(line);
			}
			free(error_line);
			break;
	}

	return retval;
}

/* Sort a list of filenames alfabeticly.
 */
static t_filelist *sort_filelist(t_filelist *filelist) {
	t_filelist *start = NULL, *newpos, *prev, *newitem;

	while (filelist != NULL) {
		newitem = filelist;
		filelist = filelist->next;

		prev = NULL;
		newpos = start;
		while (newpos != NULL) {
			if (newitem->is_dir && (newpos->is_dir == false)) {
				break;
			}
			if (newitem->is_dir == newpos->is_dir) {
				if (strcmp(newpos->name, newitem->name) >= 0) {
					break;
				}
			}
			prev = newpos;
			newpos = newpos->next;
		}

		if (prev == NULL) {
			newitem->next = start;
			start = newitem;
		} else {
			prev->next = newitem;
			newitem->next = newpos;
		}
	}
	
	return start;
}

/* free() a list of filenames.
 */
static t_filelist *remove_filelist(t_filelist *filelist) {
	t_filelist *file;

	while (filelist != NULL) {
		file = filelist;
		filelist = filelist->next;
		free(file->name);
		free(file);
	}

	return NULL;
}

/* Converts a filesize to a string.
 */
static void filesize2str(char *buffer, int fsize) {
	if (fsize < 1024) {
		sprintf(buffer, "%d byte", fsize);
	} else if (fsize < 1048576) {
		sprintf(buffer, "%0.1f kB", (double)fsize / 1024);
	} else {
		sprintf(buffer, "%0.1f MB", (double)fsize / 1048576);
	}
}

/* Read the content of a directory and send it in a formatted list to the client.
 */
int show_index(t_session *session) {
	int dir_len, total_files = 0, total_fsize = 0, retval = -1;
	char *end, size[32], *filename, *new_file, color = '0', timestr[33];
	DIR *dp;
	struct dirent *dir_info;
	struct stat status;
	struct tm s;
	t_filelist *filelist = NULL, *file;

#ifdef HAVE_COMMAND
	req_index++;
#endif

	session->mimetype = NULL;
	dir_len = strlen(session->file_on_disk);
	end = session->file_on_disk + dir_len;
	while ((*end != '/') && (end != session->file_on_disk)) {
		dir_len--;
		end--;
	}

	if (end == session->file_on_disk) {
		return 404;
	} else {
		*end = '\0';
	}

	if ((dp = opendir(session->file_on_disk)) == NULL) {
		if (session->config->ServerUid != (uid_t)-1) {
			if (errno == EACCES) {
				return 403;
			}
		}
		return 404;
	}
	if (session->config->ServerUid == (uid_t)-1) {
		switch (fs_access_allowed(session->file_on_disk, session->host->UserId, session->host->GroupId, &(session->host->Groups), directory)) {	
			case error:
				closedir(dp);
				return 500;
			case no:
				closedir(dp);
				return 403;
			case yes:
				break;
		}
	}

	if (session->host->FollowSymlinks == false) {
		switch (contains_symlink(session->file_on_disk)) {
			case error:
				return 500;
			case no:
				break;
			case yes:
				return 403;
		}
	}

	do {
		if (send_header(session, 200) == -1) {
			break;
		}
		if (session->HEAD_request == false) {
			if ((filename = new_file = (char*)malloc(dir_len + 2)) == NULL) {
				break;
			}
			memcpy(filename, session->file_on_disk, dir_len);
			*(filename + dir_len) = '/';
			dir_len++;
			*(filename + dir_len) = '\0';
			while ((dir_info = readdir(dp)) != NULL) {
				if ((dir_info->d_name[0] != '.') || (strcmp(dir_info->d_name, "..") == 0)) {
					if ((new_file = realloc(filename, dir_len + strlen(dir_info->d_name) + 1)) != NULL) {
						filename = new_file;
						strcpy(filename + dir_len, dir_info->d_name);
						stat(filename, &status);

						if ((file = (t_filelist*)malloc(sizeof(t_filelist))) != NULL) {
							file->name = strdup(dir_info->d_name);
							file->size = status.st_size;
							file->time = status.st_mtime;
							file->is_dir = S_ISDIR(status.st_mode);
							file->next = filelist;
						} else {
							remove_filelist(filelist);
							closedir(dp);
							return -1;
						}

						filelist = file;
					} else {
						break;
					}
				}
			}
			free(filename);
			if (new_file == NULL) {
				break;
			}

			filelist = sort_filelist(filelist);

			if (session->keep_alive) {
				if (send_buffer(session, chunked, 28) == -1) {
					break;
				}
			} 
			if (send_buffer(session, "Content-Type: text/html\r\n\r\n", 27) == -1) {
				break;
			} else if (send_chunk(session, "<html><head><title>Index of ", 28) == -1) {
				break;
			} else if (send_chunk(session, session->URI, strlen(session->URI)) == -1) {
				break;
			} else if (send_chunk(session,	"</title>\n"
											"<style type=\"text/css\">"
											"BODY { margin:10px ; background-color:#000030 } "
											"TD { font-family:arial ; font-size:12px } "
											"A { color:#800000 ; text-decoration:none } "
											"A:hover { color:#ff4040 ; text-decoration:underline } "
											"TR.line0 { background-color:#e0e0ff } "
											"TR.line1 { background-color:#f0f0ff } "
											".path { color:#ffffa0 ; font-family:arial ; font-size:20px ; text-align:right ; font-weight:bold ; margin-bottom:10px }"
											"</style>\n"
											"<body><div class=path>", 445) == -1) { 
				break;
			} else if (send_chunk(session, *(session->host->Hostname.item), strlen(*(session->host->Hostname.item))) == -1) {
				break;
			} else if (send_chunk(session, ":", 1) == -1) {
				break;
			} else if (send_chunk(session, session->URI, strlen(session->URI)) == -1) {
				break;
			} else if (send_chunk(session,	"</div>\n<table width=100% border=0 cellpadding=1 cellspacing=0>\n"
											"<tr bgcolor=#8080ff><td width=40></td><td><b>Filename</b></td><td width=150><b>Date</b></td><td align=right width=80><b>Size</b></td><td width=30></td></tr>\n", 220) == -1) {
				break;
			} 
			while (filelist != NULL) {
				if (send_chunk(session, "<tr class=line", 14) == -1) {
					break;
				} else if (send_chunk(session, &color, 1) == -1) {
					break;
				} else if (send_chunk(session, "><td>", 5) == -1) {
					break;
				}
				if (filelist->is_dir) {
					if (send_chunk(session, "&nbsp;[dir]", 11) == -1) {
						break;
					}
				}
				if (send_chunk(session, "</td><td><a href=\"", 18) == -1) {
					break;
				} else if (send_chunk(session, session->URI, strlen(session->URI)) == -1) {
					break;
				} else if (send_chunk(session, filelist->name, strlen(filelist->name)) == -1) {
					break;
				}
				if (filelist->is_dir) {
					if (send_chunk(session, "/", 1) == -1) {
						break;
					}
				}
				if (send_chunk(session, "\">", 2) == -1) {
					break;
				} else if (send_chunk(session, filelist->name, strlen(filelist->name)) == -1) {
					break;
				}
				if (filelist->is_dir) {
					if (send_chunk(session, "/", 1) == -1) {
						break;
					}
				}
				if (send_chunk(session, "</a></td><td>", 13) == -1) {
					break;
				}
				s = *localtime(&(filelist->time));
				*(timestr + 32) = '\0';
				strftime(timestr, 32, "%d %b %Y, %X", &s);
				if (send_chunk(session, timestr, strlen(timestr)) == -1) {
					break;
				} else if (send_chunk(session, "</td><td align=right>", 21) == -1) {
					break;
				}
				if (filelist->is_dir == false) {
					filesize2str(size, filelist->size);
					if (send_chunk(session, size, strlen(size)) == -1) {
						break;
					}
				}
				if (send_chunk(session, "</td><td></td></tr>\n", 20) == -1) {
					break;
				}
				color = '0' + (1 - (color - '0'));

				total_files++;
				total_fsize += filelist->size;
				filelist = filelist->next;
			}
			if (filelist != NULL) {
				break;
			}

			if (send_chunk(session,	"<tr bgcolor=#8080ff><td></td><td><b>", 36) == -1) {
				break;
			}
			sprintf(size, "%d", total_files);
			if (send_chunk(session, size, strlen(size)) == -1) {
				break;
			} else if (send_chunk(session, " files</b></td><td></td><td align=right><b>", 43) == -1) {
				break;
			}
			filesize2str(size, total_fsize);
			if (send_chunk(session, size, strlen(size)) == -1) {
				break;
			} else if (send_chunk(session, "</b></td><td></td></tr>\n</table></body></html>\n", 46) == -1) {
				break;
			} else if (send_chunk(session, NULL, 0) == -1) {
				break;
			}
		} else {
			send_buffer(session, "\r\n", 2);
		}

		retval = 200;
	} while (false);

	remove_filelist(filelist);
	closedir(dp);

	return retval;
}

/* Send the result of an OPTIONS request method to the client.
 */
int handle_options_request(t_session *session) {
	int code = 200, handle;

	if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
		if (session->URI_is_dir) {
			code = 501;
		} else {
			code = 404;
		}
		if (session->config->ServerUid != (uid_t)-1) {
			if (errno == EACCES) {
				code = 403;
			}
		}
	} else {
		close(handle);

		if (session->config->ServerUid == (uid_t)-1) {
			switch (fs_access_allowed(session->file_on_disk, session->host->UserId, session->host->GroupId, &(session->host->Groups), file)) {
				case error:
					code = 500;
					break;
				case no:
					code = 403;
					break;
				case yes:
					break;
			}
		}
	}
	if (session->host->FollowSymlinks == false) {
		switch (contains_symlink(session->file_on_disk)) {
			case error:
				code = 500;
				break;
			case no:
				break;
			case yes:
				code = 403;
		}
	}

	if (send_header(session, code) == -1) {
		code = -1;
	}
	if (code == 200) {
		if ((code = send_buffer(session, hs_allow, 25)) != -1) {
			if (session->is_CGI_script) {
				code = send_buffer(session, hs_post, 6);
			}
			if (code != -1) {
				code = send_buffer(session, hs_eol, 2);
			}
		}
	}
	if (code != -1) {
		if ((code = send_buffer(session, hs_conlen, 16)) != -1) {
			code = send_buffer(session, "0\r\n\r\n", 5);
		}   
	}

	if (code == -1) {
		return -1;
	} else {
		return 200;
	}
}

int handle_trace_request(t_session *session) {
	int result = -1, code, body_size, len;
	char buffer[MAX_TRACE_HEADER + 2];
	t_headerfield *header;

	body_size = 3;
	body_size += strlen(session->method) + strlen(session->URI);
	if (session->vars != NULL) {
		body_size += 1 + strlen(session->vars);
	}
	body_size += strlen(session->http_version);

	header = session->headerfields;
	while (header != NULL) {
		body_size += strlen(header->data) + 1;
		header = header->next;
	}

	buffer[MAX_TRACE_HEADER + 1] = '\0';

	do {
		// Header
		if (sprintf(buffer, "%d\r\nContent-Type: message/http\r\n\r\n", body_size) < 0) {
			break;
		} else if (send_header(session, 200) == -1) {
			break;
		} else if (send_buffer(session, hs_conlen, 16) == -1) {
			break;
		} else if (send_buffer(session, buffer, strlen(buffer)) == -1) {
			break;
		}

		// Body
		if ((code = snprintf(buffer, MAX_TRACE_HEADER, "%s %s", session->method, session->URI)) < 0) {
			break;
		} else if (code >= MAX_TRACE_HEADER) {
			break;
		} else if (session->vars != NULL) {
			if ((code = snprintf(buffer + strlen(buffer), MAX_TRACE_HEADER, "?%s", session->vars)) < 0) {
				break;
			} else if (code >= MAX_TRACE_HEADER) {
				break;
			}
		}
		if ((code = snprintf(buffer + strlen(buffer), MAX_TRACE_HEADER, " %s\r\n", session->http_version)) < 0) {
			break;
		} else if (send_buffer(session, buffer, strlen(buffer)) == -1) {
			break;
		}

		header = session->headerfields;
		while (header != NULL) {
			len = strlen(header->data);
			*(header->data + len) = '\n';
			if (send_buffer(session, header->data, len + 1) == -1) {
				*(header->data + len) = '\0';
				result = -2;
				break;
			}
			*(header->data + len) = '\0';
			header = header->next;
		}
		if (result == -2) {
			break;
		}
	
		result = 200;
	} while (false);

	return result;
}

#ifdef HAVE_PLUGIN
/* Give control to the plugin.
 */
int handle_plugin(t_session *session) {
	t_request_info *request_info;

	if ((request_info = (t_request_info*)malloc(sizeof(t_request_info))) != NULL) {
		request_info->fd = session->client_socket;
		request_info->fp = fdopen(session->client_socket, "w");
		request_info->keep_alive = session->keep_alive;
		request_info->parameters = session->vars;
		request_info->cookies = get_headerfield("Cookie:", session->headerfields);

		if (send_header(session, 200) == -1) {
			return 500;
		}
		chdir(session->host->WebsiteRoot);
		plugin_request(request_info);

		fflush(request_info->fp);
		fclose(request_info->fp);
		free(request_info);

		return 200;
	} else {
		return 500;
	}
}
#endif
