#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "client.h"
#include "log.h"

#define TIMER_OFF -1

typedef struct type_client {
	t_session *session;
	int remove_delay;

	struct type_client *next;
} t_client;

typedef struct type_banned {
	long ip;
	int  timer;
	int  bantime;
	unsigned long connect_attempts;
	
	struct type_banned *next;
} t_banned;

t_client *clientlist;
t_banned *banlist;
pthread_mutex_t client_mutex;
pthread_mutex_t ban_mutex;

static void insecure_kick_ip(long ip);

/* Initialize this module.
 */
void init_clientmodule(void) {
	clientlist = NULL;
	banlist = NULL;
	pthread_mutex_init(&client_mutex, NULL);
	pthread_mutex_init(&ban_mutex, NULL);
}

/* Add the session record of a client to the clientlist.
 */
int add_client(t_session *session) {
	t_client *new;

	if ((new = (t_client*)malloc(sizeof(t_client))) != NULL) {
		new->session = session;
		new->remove_delay = -1;

		pthread_mutex_lock(&client_mutex);

		new->next = clientlist;
		clientlist = new;

		pthread_mutex_unlock(&client_mutex);

		return 0;
	} else {
		return -1;
	}
}

/* Remember the clientrecord for flooding prevention
 */
void mark_client_for_removal(t_session *session, int delay) {
	t_client *list;

	pthread_mutex_lock(&client_mutex);

	list = clientlist;
	while (list != NULL) {
		if (list->session == session) {
			close_socket(list->session);
			list->remove_delay = delay - 1;
			break;
		}
		list = list->next;
	}

	pthread_mutex_unlock(&client_mutex);
}

/* Check the remove_delay timers and remove client
 * when timer has reached 0
 */
void check_remove_delay_timers(void) {
	t_client *list, *prev = NULL, *next;

	pthread_mutex_lock(&client_mutex);

	list = clientlist;
	while (list != NULL) {
		next = list->next;
		switch (list->remove_delay) {
			case -1:
				break;
			case 0:
				free(list->session);
				free(list);
				list = prev;
				if (prev == NULL) {
					clientlist = next;
				} else {
					prev->next = next;
				}
				break;
			default:
				list->remove_delay--;
		}
		prev = list;
		list = next;
	}

	pthread_mutex_unlock(&client_mutex);
}

/* Remove a client from the clientlist.
 */
void remove_client(t_session *session, bool free_session) {
	t_client *deadone = NULL, *list;

	pthread_mutex_lock(&client_mutex);

	if (clientlist != NULL) {
		if (clientlist->session == session) {
			deadone = clientlist;
			clientlist = clientlist->next;
		} else {
			list = clientlist;
			while (list->next != NULL) {
				if (list->next->session == session) {
					deadone = list->next;
					list->next = deadone->next;
					break;
				}
				list = list->next;
			}
		}
	}

	pthread_mutex_unlock(&client_mutex);

	if (deadone != NULL) {
		if (free_session) {
			close_socket(deadone->session);
			free(deadone->session);
		}
		free(deadone);
	}
}

/* Check whether to allow or deny a new connection.
 */
int connection_allowed(long ip, int maxperip, int maxtotal) {
	bool banned = false;
	t_banned *ban;
	int perip = 0, total = 0;
	t_client *client;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->connect_attempts++;
			banned = true;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);

	if (banned) {
		return ca_BANNED;
	}

	pthread_mutex_lock(&client_mutex);

	client = clientlist;
	while (client != NULL) {
		if (client->session->ip_address == ip) {
			perip++;
		}
		total++;
		client = client->next;
	}

	pthread_mutex_unlock(&client_mutex);

	if (perip < maxperip) {
		if (total < maxtotal) {
			return total;
		} else {
			return ca_TOOMUCH_TOTAL;
		}
	} else {
		return ca_TOOMUCH_PERIP;
	}
}

/* Disconnect all connected clients.
 */
void disconnect_clients(t_config *config) {
	t_client *client;
	t_directory *dir;
	int max_wait = 30;

	pthread_mutex_lock(&client_mutex);

	client = clientlist;
	while (client != NULL) {
		client->session->force_quit = true;
		client = client->next;
	}

	pthread_mutex_unlock(&client_mutex);

	while ((clientlist != NULL) && (max_wait-- > 0)) {
		usleep(100000);
	}

	dir = config->directory;
	while (dir != NULL) {
		dir->nr_of_clients = 0;
		dir = dir->next;
	}
}

/* Check the request sending speed of the clients.
 */
void check_flooding(t_config *config) {
	t_client *client;

	if (config->ban_on_flooding > 0) {
		pthread_mutex_lock(&client_mutex);

		client = clientlist;
		while (client != NULL) {
			if (client->session->flooding_timer > 0) {
				client->session->flooding_timer--;
			} else if (client->session->flooding_counter < config->flooding_count) {
				client->session->flooding_timer = config->flooding_time;
				client->session->flooding_counter = 0;
			} else if (ip_allowed(client->session->ip_address, config->banlist_mask)) {
				ban_ip(client->session->ip_address, config->ban_on_flooding, false);
				log_warning(config->system_logfile, "Client banned because of flooding", client->session->ip_address);
				if (config->kick_on_ban) {
					insecure_kick_ip(client->session->ip_address);
				} else {
					close_socket(client->session);
				}
			}
			client = client->next;
		}

		pthread_mutex_unlock(&client_mutex);
	}
}

#ifdef HAVE_COMMAND
/* Extra client functions
 */
void print_clientlist(FILE *fp) {
	t_client *client;
	char *host;
	unsigned char *ip_address;

	pthread_mutex_lock(&client_mutex);

	client = clientlist;
	while (client != NULL) {
		fprintf(fp, "session id  : %d\n", client->session->client_id);
		ip_address = (unsigned char*)&(client->session->ip_address);
		fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
		fprintf(fp, "socket      : %d\n", client->session->client_socket);
		fprintf(fp, "Kept alive  : %d\n", client->session->kept_alive);
		if (client->session->remote_user != NULL) {
		fprintf(fp, "Remote user : %s\n", client->session->remote_user);
		}
		if (client->session->file_on_disk != NULL) {
			fprintf(fp, "File on disk: %s\n", client->session->file_on_disk);
			if ((host = get_headerfield("Host:", client->session->headerfields)) != NULL) {
				fprintf(fp, "Host        : %s\n", host);
			}
		}
		client = client->next;
		if (client != NULL) {
			fprintf(fp, "---\n");
		}
	}

	pthread_mutex_unlock(&client_mutex);
}
#endif

/* Disconnect a client.
 */
void kick_client(int id) {
	t_client *client;
	
	pthread_mutex_lock(&client_mutex);

	client = clientlist;
	while (client != NULL) {
		if (client->session->client_id == id) {
			client->session->force_quit = true;
			break;
		}
		client = client->next;
	}

	pthread_mutex_unlock(&client_mutex);
}

/* Kick an IP address.
 */
static void insecure_kick_ip(long ip) {
	t_client *client;

	client = clientlist;
	while (client != NULL) {
		if (client->session->ip_address == ip) {
			client->session->force_quit = true;
		}
		client = client->next;
	}
}
void kick_ip(long ip) {
	pthread_mutex_lock(&client_mutex);

	insecure_kick_ip(ip);

	pthread_mutex_unlock(&client_mutex);
}

/* IP ban functions
 */
int ban_ip(long ip, int timer, bool kick_on_ban) {
	int retval = 0;
	t_banned *ban;
	bool new_ip = true;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = timer;
			ban->bantime = timer;
			new_ip = false;
			break;
		}
		ban = ban->next;
	}

	if (new_ip) {
		if ((ban = (t_banned*)malloc(sizeof(t_banned))) != NULL) {
			ban->ip = ip;
			ban->timer = timer;
			ban->bantime = timer;
			ban->connect_attempts = 0;
			ban->next = banlist;

			banlist = ban;
		} else {
			retval = -1;
		}
	}

	pthread_mutex_unlock(&ban_mutex);

	if (kick_on_ban) {
		kick_ip(ip);
	}

	return retval;
}

/* Reset the timer of a banned IP address.
 */
void reban_ip(long ip) {
	t_banned *ban;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		if (ban->ip == ip) {
			ban->timer = ban->bantime;
			break;
		}
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

/* Check the timers of the banlist.
 */
void check_banlist(t_config *config) {
	t_banned *ban, *prev = NULL, *next;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		next = ban->next;
		switch (ban->timer) {
			case TIMER_OFF:
				break;
			case 0:
				if (prev == NULL) {
					banlist = next;
				} else {
					prev->next = next;
				}
				log_unban(config->system_logfile, ban->ip, ban->connect_attempts);
				free(ban);
				ban = prev;
				break;
			default:
				ban->timer--;
		}
		prev = ban;
		ban = next;
	}
	
	pthread_mutex_unlock(&ban_mutex);
}

/* Unban an IP address.
 */
void unban_ip(long ip) {
	t_banned *ban, *prev = NULL;

	pthread_mutex_lock(&ban_mutex);
	
	ban = banlist;
	while (ban != NULL) {
		if ((ban->ip == ip) || (ip == 0)) {
			if (prev == NULL) {
				banlist = ban->next;
			} else {
				prev->next = ban->next;
			}
			free(ban);
			break;
		}
		prev = ban;
		ban = ban->next;
	}

	pthread_mutex_unlock(&ban_mutex);
}

#ifdef HAVE_COMMAND
/* Print the list of banned IP addresses.
 */
void print_banlist(FILE *fp) {
	t_banned *ban;
	unsigned char *ip_address;

	pthread_mutex_lock(&ban_mutex);

	ban = banlist;
	while (ban != NULL) {
		ip_address = (unsigned char*)&(ban->ip);
		fprintf(fp, "IP-address  : %hhu.%hhu.%hhu.%hhu\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
		fprintf(fp, "seconds left: %d\n", ban->timer);
		ban = ban->next;
		if (ban != NULL) {
			fprintf(fp, "---\n");
		}
	}

	pthread_mutex_unlock(&ban_mutex);
}
#endif
