/* Hiawatha | command.c
 *
 * CommandChannel routines
 */
#include "config.h"

#ifdef HAVE_COMMAND

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif
#ifdef HAVE_RPCSVC_CRYPT_H
#include <rpcsvc/crypt.h>
#endif
#include "command.h"
#include "libstr.h"
#include "client.h"
#include "session.h"

#define MAX_IDLE_TIME 30

t_admin *adminlist, *loop, *next;
static char *prompt = "cmd> ";
static char *unknown_cmd = "Unknown command. Type 'help' for the commandlist.\n";

volatile int cl_lock = 0;
pthread_mutex_t command_mutex;

unsigned long clients_served = 0, req_files = 0, req_cgi = 0, req_index = 0;

/* A client needs to change a session record. Admins must stay out.
 */
void CC_client_lock(void) {
	pthread_mutex_lock(&command_mutex);

	while (cl_lock < 0) {
		usleep(1000);
	}
	cl_lock++;
	
	pthread_mutex_unlock(&command_mutex);
}

/* Client is done changing the session record.
 */
void CC_client_unlock(void) {
	cl_lock--;
}

/* Admin wants to read a session record. nr_of_clients must stay out.
 */
void CC_admin_lock(void) {
	pthread_mutex_lock(&command_mutex);

	while (cl_lock > 0) {
		usleep(1000);
	}
	cl_lock--;
	
	pthread_mutex_unlock(&command_mutex);
}

/* Admin is done reading a session record.
 */
void CC_admin_unlock(void) {
	cl_lock++;
}

/* Initialize the CommandChannel
 */
void init_commandmodule(void) {
    adminlist = NULL;
	loop = NULL;
	next = NULL;
	pthread_mutex_init(&command_mutex, NULL);
}

/* An administrator has connected to the CommandChannel.
 */
int add_admin(int sock) {
	t_admin *new;

	if ((new = (t_admin*)malloc(sizeof(t_admin))) != NULL) {
		if ((new->fp = fdopen(sock, "r+")) != NULL) {
			fprintf(new->fp, "\nHiawatha v"VERSION" CommandChannel\nPassword: ");
			fflush(new->fp);
			new->next = adminlist;
			new->socket = sock;
			new->allowed = false;
			new->timer = MAX_IDLE_TIME;
			adminlist = new;

			return 0;
		} else {
			return -1;
		}
	} else {
		return -1;
	}
}

/* An administrator has left the CommandChannel
 */
void remove_admin(int sock) {
	t_admin *deadone = NULL, *check;

	if (adminlist != NULL) {
		if (adminlist->socket == sock) {
			deadone = adminlist;
			adminlist = adminlist->next;
		} else {
			check = adminlist;
			while (check->next != NULL) {
				if (check->next->socket == sock) {
					deadone = check->next;
					check->next = deadone->next;
					break;
				}
				check = check->next;
			}
		}
	}

	if (deadone != NULL) {
		fclose(deadone->fp);
		free(deadone);
	}
}

/* Disconnect al the administrators.
 */
void disconnect_admins(void) {
	t_admin *admin;

	while (adminlist != NULL) {
		admin = adminlist;
		adminlist = adminlist->next;

		close(admin->socket);
		free(admin);
	}
}

/* Return the first admin record.
 */
t_admin *first_admin(void) {
	if ((loop = adminlist) != NULL) {
		next = loop->next;
		return loop;
	} else {
		return (next = NULL);
	}
}

/* Return the next admin record.
 */
t_admin *next_admin(void) {
	if ((loop = next) != NULL) {
		next = loop->next;
		return loop;
	} else {
		return NULL;
	}
}

/* Check administratos (only auto-logout timers for now).
 */
void check_adminlist(void) {
	t_admin *admin, *prev_admin = NULL, *next_admin;

	admin = adminlist;
	while (admin != NULL) {
		next_admin = admin->next;
		if (admin->timer == 0) {
			close(admin->socket);
			if (prev_admin == NULL) {
				adminlist = next_admin;
			} else {
				prev_admin->next = next_admin;
			}
			free(admin);
			admin = prev_admin;
		} else {
			admin->timer--;
		}
		prev_admin = admin;
		admin = next_admin;
	}
}

/* Show help info.
 */
static void show_help(FILE *fp) {
	fprintf(fp,	"ban <ip>[,<time>]  : ban ip-address (for <time> seconds)\n"
				"banlist            : show all banned ip-addresses\n"
				"clientlist         : show connected clients\n"
				"kickid <client-id> : disconnect client <client-id>\n"
				"kickip <ip-address>: disconnect clients with that ip-address\n"
				"kickall            : disconnect all clients\n"
				"shutdown           : shutdown the webserver\n"
				"status             : status of the webserver\n"
				"quit               : quit CommandChannel\n"
				"unban <ip>         : unban ip-address\n");
}

/* Handle a administrator command.
 */
int handle_admin(t_admin *admin, t_config *config) {
	int retval = 0, id, time;
	char *line, *cmd, *param, *param2, *encrypted, salt[3];
	unsigned long ip;

	if ((line = (char*)malloc(65)) != NULL) {
		if (fgets(line, 64, admin->fp) != NULL) {
			*(line + 64) = '\0';
			admin->timer = MAX_IDLE_TIME;
			if (admin->allowed) {
				if (split_string(line, &cmd, &param, ' ') == 0) {
					if (strcmp(cmd, "kickid") == 0) {
						if ((id = str2int(param)) != -1) {
							kick_client(id);
						}
					} else if (strcmp(cmd, "kickip") == 0) {
						if (parse_ip(param, &ip) != -1) {
							kick_ip(ip);
						}
					} else if (strcmp(cmd, "ban") == 0) {
						if (split_string(param, &param, &param2, ',') == 0) {
							time = str2int(param2);
						} else {
							time = TIMER_OFF;
						}
						if (parse_ip(param, &ip) != -1) {
							ban_ip(ip, time, false);
						}
					} else if (strcmp(cmd, "unban") == 0) {
						if (strcmp(param, "all") == 0) {
							retval = cc_UNBAN_CLIENTS;
						} else if (parse_ip(param, &ip) != -1) {
							unban_ip(ip);
						}
					} else {
						fprintf(admin->fp, unknown_cmd);
					}
				} else {
					cmd = remove_spaces(line);
					if (strcmp(cmd, "quit") == 0) {
						retval = cc_DISCONNECT;
					} else if (strcmp(cmd, "help") == 0) {
						show_help(admin->fp);
					} else if (strcmp(cmd, "shutdown") == 0) {
						retval = cc_SHUTDOWN;
					} else if (strcmp(cmd, "clientlist") == 0) {
						CC_admin_lock();
						print_clientlist(admin->fp);
						CC_admin_unlock();
					} else if (strcmp(cmd, "banlist") == 0) {
						print_banlist(admin->fp);
					} else if (strcmp(cmd, "kickall") == 0) {
						disconnect_clients(config);
					} else if (strcmp(cmd, "status") == 0) {
						fprintf(admin->fp, "nr_of_clients served   : %lu\n", clients_served);
						fprintf(admin->fp, "Files requested  : %lu\n", req_files);
						fprintf(admin->fp, "CGI's requested  : %lu\n", req_cgi);
						fprintf(admin->fp, "Indexes requested: %lu\n", req_index);
					} else {
						fprintf(admin->fp, unknown_cmd);
					}
				}
			} else {
				memcpy(salt, config->command_port->name, 2);
				salt[2] = '\0';
				encrypted = crypt(remove_spaces(line), salt);
				if ((admin->allowed = (strcmp(encrypted, config->command_port->name) == 0)) == false) {
					retval = cc_DISCONNECT;
				}
			}
		} else {
			retval = cc_DISCONNECT;
		}
		free(line);

		if (retval >= 0) {
			fprintf(admin->fp, "----------\n");
			fprintf(admin->fp, prompt);
			fflush(admin->fp);
		}
	} else {
		retval = cc_DISCONNECT;
	}

	return retval;
}

#endif
