#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include "session.h"
#include "libcgi.h"
#include "libstr.h"
#include "session.h"

int fork_CGI_process(t_session *session, t_cgi_info *cgi_info) {
	int cgi_pid, post_pipe[2], html_pipe[2], error_pipe[2], i;
	char *slash, *run[7], cgi_time[16];

	do {
		if (pipe(post_pipe) != -1) {
			if (pipe(html_pipe) != -1) {
				if (pipe(error_pipe) != -1) {
					break;
				}
				close(html_pipe[0]);
				close(html_pipe[1]);
			}
			close(post_pipe[0]);
			close(post_pipe[1]);
		}
		return -1;
	} while (false);

	switch (cgi_pid = fork()) {
		case -1:
			break;
		case 0:
			/* Child, executes CGI program.
			 */
			dup2(post_pipe[0], STDIN_FILENO);
			dup2(html_pipe[1], STDOUT_FILENO);
			dup2(error_pipe[1], STDERR_FILENO);

			close(post_pipe[0]);
			close(post_pipe[1]);
			close(html_pipe[0]);
			close(html_pipe[1]);
			close(error_pipe[0]);
			close(error_pipe[1]);

			fcntl(0, F_SETFD, 0);
			fcntl(1, F_SETFD, 0);
			fcntl(2, F_SETFD, 0);

			set_environment(session, NULL, 0);
			if ((slash = strrchr(session->file_on_disk, '/')) != NULL) {
				*slash = '\0';
				chdir(session->file_on_disk);
				*slash = '/';
			}

			i = 0;
			if (cgi_info->wrap_cgi) {
				run[i++] = session->config->cgi_wrapper;
				if (session->host->cgi_wrap_id != NULL) {
					setenv("CGIWRAP_ID", session->host->cgi_wrap_id, 1);
				} else {
					setenv("CGIWRAP_ID", session->local_user, 1);
				}
				if (session->host->follow_symlinks) {
					setenv("CGIWRAP_FOLLOWSYMLINKS", "true", 1);
				} else {
					setenv("CGIWRAP_FOLLOWSYMLINKS", "false", 1);
				}
				cgi_time[15] = '\0';
				snprintf(cgi_time, 15, "%d", session->config->time_for_cgi);
				setenv("CGIWRAP_TIMEFORCGI", cgi_time, 1);
			} else if (setsid() == -1) {
				exit(-1);
			}
			if (session->cgi_handler != NULL) {
				run[i++] = session->cgi_handler;
				run[i++] = session->cgi_handler;
			} else {
				if (cgi_info->wrap_cgi) {
					run[i++] = "-";
				}
				run[i++] = session->file_on_disk;
			}
			run[i++] = session->file_on_disk;
			run[i] = NULL;

			execvp(run[0], run + 1);
			exit(-1);
		default:
			/* Parent, reads CGI output
			 */
			close(post_pipe[0]);
			close(error_pipe[1]);
			close(html_pipe[1]);

			cgi_info->to_cgi = post_pipe[1];
			cgi_info->from_cgi = html_pipe[0];
			cgi_info->cgi_error = error_pipe[0];

			if (cgi_info->from_cgi > cgi_info->cgi_error) {
				cgi_info->highest_fd = cgi_info->from_cgi + 1;
			} else {
				cgi_info->highest_fd = cgi_info->cgi_error + 1;
			}

			/* Send POST data to CGI program.
			 */
			if ((session->body != NULL) && (session->content_length > 0)) {
				write(cgi_info->to_cgi, session->body, session->content_length);
			}
	}

	return cgi_pid;
}

int read_from_CGI_process(t_session *session, t_cgi_info *cgi_info) {
	bool read_again;
	fd_set read_fds;
	int bytes_read;

	cgi_info->input_result = cgi_info->error_result = rc_NOTHING;

	do {
		read_again = false;

		FD_ZERO(&read_fds);
		FD_SET(cgi_info->from_cgi, &read_fds);
		FD_SET(cgi_info->cgi_error, &read_fds);

		switch (select(cgi_info->highest_fd, &read_fds, NULL, NULL, &(cgi_info->select_timeout))) {
			case -1:
				return rc_SELECT_ERROR;
			case 0:
				if (session->force_quit) {
					return rc_FORCE_QUIT;
				} else if (--(cgi_info->timer) <= 0) {
					return rc_TIMEOUT;
				} else {
					cgi_info->select_timeout.tv_sec = 1;
					cgi_info->select_timeout.tv_usec = 0;
					read_again = true;
				}
		}
	} while (read_again);

	do {
		read_again = false;

		if (FD_ISSET(cgi_info->from_cgi, &read_fds)) {
			bytes_read = read(cgi_info->from_cgi, cgi_info->input_buffer + cgi_info->input_len, cgi_info->input_buffer_size - cgi_info->input_len);
			if (bytes_read == -1) {
				if (errno != EINTR) {
					cgi_info->input_result = rc_READ_ERROR;
				} else {
					read_again = true;
				}
			} else if (bytes_read == 0) {
				cgi_info->input_result = rc_END_OF_DATA;
			} else if (bytes_read > 0) {
				cgi_info->input_result = rc_READ_DATA;
				cgi_info->input_len += bytes_read;
			}
		}
	} while (read_again);

	do {
		read_again = false;

		if (FD_ISSET(cgi_info->cgi_error, &read_fds)) {
			bytes_read = read(cgi_info->cgi_error, cgi_info->error_buffer + cgi_info->error_len, cgi_info->error_buffer_size - cgi_info->error_len);
			if (bytes_read == -1) {
				if (errno != EINTR) {
					cgi_info->error_result = rc_READ_ERROR;
				} else {
					read_again = true;
				}
			} else if (bytes_read == 0) {
				cgi_info->error_result = rc_END_OF_DATA;
			} else if (bytes_read > 0) {
				cgi_info->error_result = rc_READ_DATA;
				cgi_info->error_len += bytes_read;
			}
		}
	} while (read_again);

	return rc_OKE;
}
