#include "config.h"
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <time.h>
#include <pthread.h>
#include <sys/file.h>
#include "log.h"

#define BUFFER_SIZE 2000

pthread_mutex_t accesslog_mutex;

void init_logmodule(void) {
	pthread_mutex_init(&accesslog_mutex, NULL);
}


/* Write a timestamp to a logfile.
 */
static void print_timestamp(char *str) {
	time_t t;
	struct tm *s;

	time(&t);
	s = localtime(&t);
	strftime(str, 60, "%a %d %b %Y %X|", s);
}

/* Write an IP address to a logfile.
 */
static void print_ip_address(char *str, long ip_address) {
	unsigned char *ip;

	ip = (unsigned char*)&ip_address;
	sprintf(str, "%hhu.%hhu.%hhu.%hhu|", ip[0], ip[1], ip[2], ip[3]);
}

static char *secure_string(char *str) {
	char *c = str;

	while (*c != '\0') {
		if (*c == '\27') {
			*c = ' ';
		}
		c++;
	}

	return str;
}

// ---< Main log functions >-------------

/* Log a text.
 */
void log_string(char *logfile, char *data) {
	FILE *fp;
	char str[60];

	if ((fp = fopen(logfile, "a")) != NULL) {
		print_timestamp(str);
		fprintf(fp, "%s%s\n", str, data);
		fclose(fp);
	}
}

/* Log a CGI error.
 */
void log_CGI_error(char *logfile, char *CGI_file, char *error) {
	FILE *fp;
	char *c, str[60];

	c = error;
	while (*c != '\0') {
		if (*c == '\n') {
			*c = '|';
		}
		c++;
	}

	if ((fp = fopen(logfile, "a")) != NULL) {
		print_timestamp(str);
		fprintf(fp, "%s%s|%s\n", str, CGI_file, secure_string(error));
		fclose(fp);
	}
}

/* Log a warning.
 */
void log_warning(char *logfile, char *data, long ip_address) {
	FILE *fp;
	char str[60];

	if ((fp = fopen(logfile, "a")) != NULL) {
		print_ip_address(str, ip_address);
		print_timestamp(str + strlen(str));
		fprintf(fp, "%s%s\n", str, data);
		fclose(fp);
	}
}

/* Log a HTTP request.
 */
void log_request(t_session *session, int code) {
	char rcode[4], *user;
	t_headerfield *headerfield;
	bool should_log;
	char str[BUFFER_SIZE + 1];
	int offset;

	if (ip_allowed(session->ip_address, session->config->logfile_mask) != deny) {
		str[BUFFER_SIZE] = '\0';

		sprintf(rcode, "%d", code);
		if ((user = session->remote_user) == NULL) {
			user = "";
		}

		print_ip_address(str, session->ip_address);
		offset = strlen(str);
		print_timestamp(str + offset);
		offset += strlen(str + offset);

		snprintf(str + offset, BUFFER_SIZE - offset, "%s%s|%ld|%s|%s %s", rcode, session->reason_for_403, session->bytes_sent, secure_string(user), secure_string(session->method), secure_string(session->uri));
		offset += strlen(str + offset);

		if ((offset < BUFFER_SIZE) && (session->vars != NULL)) {
			snprintf(str + offset, BUFFER_SIZE - offset, "?%s", secure_string(session->vars));
			offset += strlen(str + offset);
		}

		if (offset < BUFFER_SIZE) {
			snprintf(str + offset, BUFFER_SIZE - offset, " %s", secure_string(session->http_version));
			offset += strlen(str + offset);
		}
		
		if (offset < BUFFER_SIZE) {
			headerfield = session->headerfields;
			while (headerfield != NULL) {
				should_log = true;
				if (strlen(headerfield->data) >= 14) {
					if (memcmp("Authorization:", headerfield->data, 14) == 0) {
						should_log = false;
					}
				}
				if (should_log) {
					snprintf(str + offset, BUFFER_SIZE - offset, "|%s", secure_string(headerfield->data));
					if ((offset += strlen(str + offset)) >= BUFFER_SIZE) {
						break;
					}
				}
				headerfield = headerfield->next;
			}
		}

		pthread_mutex_lock(&accesslog_mutex);
		if (*(session->host->access_fp) == NULL) {
			*(session->host->access_fp) = fopen(session->host->access_logfile, "a");
		}
		if (*(session->host->access_fp) != NULL) {
			fprintf(*(session->host->access_fp), "%s\n", str);
		}
		pthread_mutex_unlock(&accesslog_mutex);
	}
}

/* Log garbage sent by a client.
 */
void log_garbage(t_session *session) {
	FILE *fp;
	char str[100];

	if ((session->config->garbage_logfile != NULL) && (session->request != NULL)) {
		if ((fp = fopen(session->config->garbage_logfile, "a")) != NULL) {
			print_ip_address(str, session->ip_address);
			print_timestamp(str + strlen(str));
			fprintf(fp, "%s%s\n", str, session->request);
			fclose(fp);
		}
	}
}

/* Log an unbanning.
 */
void log_unban(char *logfile, unsigned long ip_address, unsigned long connect_attempts) {
	FILE *fp;
	char str[60];

	if ((fp = fopen(logfile, "a")) != NULL) {
		print_ip_address(str, ip_address);
		print_timestamp(str + strlen(str));
		fprintf(fp, "%sUnbanned (%ld connect attempts during ban)\n", str, connect_attempts);
		fclose(fp);
	}
}

/* Close open accesslogfiles.
 */
void log_close(t_host *host) {
	pthread_mutex_lock(&accesslog_mutex);
	while (host != NULL) {
		if (*(host->access_fp) != NULL) {
			fclose(*(host->access_fp));
			*(host->access_fp) = NULL;
		}
		host = host->next;
	}
	pthread_mutex_unlock(&accesslog_mutex);
}

#ifdef DEBUG
void log_error(char *logfile, int code) {
	FILE *fp;
	char str[1024];

	if ((fp = fopen(logfile, "a")) != NULL) {
		print_timestamp(str);
		sprintf(str + strlen(str), "ERROR: ");
		strerror_r(code, str + strlen(str), 255);
		fprintf(fp, "%s\n", str);
		fclose(fp);
	}
}
#endif
