#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include "serverconfig.h"
#include "libstr.h"

#define mc_INCLUDE       -1
#define mc_READ_ERROR    -2
#define mc_DIR_CONFLICT  -3
#define mc_NO_HOSTNAME   -4
#define mc_NO_BINDING    -5
#define mc_REQUIRE       -6

#define id_NOBODY 65534

enum t_section { syntax_error = -1, none, binding, virtualhost, directory };
enum t_section_type { tag, bracket, end };

bool including = false;
t_keyvalue *variables = NULL;

static t_host *new_host(void) {
	t_host *host;

	if ((host = (t_host*)malloc(sizeof(t_host))) != NULL) {
		host->website_root       = WEBROOT_DIR;
		host->start_file         = "index.html";
		host->error_handler      = NULL;
		host->return_code        = -1;
		host->access_logfile     = LOG_DIR"/access.log";
		host->access_fileptr     = NULL;
		host->access_fp          = &(host->access_fileptr);
		host->error_logfile      = LOG_DIR"/error.log";
		init_charlist(&(host->hostname));
		host->user_websites      = false;
		host->execute_cgi        = false;
		host->show_index         = false;
		host->access_list        = NULL;
#ifdef HAVE_PLUGIN
		host->plugin_active      = false;
#endif
		host->login_message      = "Private page";
		host->passwordfile       = NULL;
		host->groupfile          = NULL;
		host->index_style        = NULL;
		host->deny_bot           = NULL;
		init_charlist(&(host->required_binding));
		init_charlist(&(host->required_group));
		host->cgi_wrap_id        = NULL;
		host->groups.number      = 0;
		host->groups.array       = NULL;
		init_charlist(&(host->volatile_object));
		init_charlist(&(host->image_referer));
		host->imgref_replacement = NULL;
		host->envir_str          = NULL;
		host->alias              = NULL;
#ifdef HAVE_SSL
		host->require_ssl        = false;
#endif
		host->prevent_cmdi       = false;
		host->prevent_sqli       = false;
		host->prevent_xss        = false;
		host->follow_symlinks    = false;

		host->next               = NULL;
	}

	return host;
}

static t_directory *new_directory(void) {
	t_directory *directory;

	if ((directory = (t_directory*)malloc(sizeof(t_directory))) != NULL) {
		directory->path               = NULL;
		directory->cgi_wrap_id        = NULL;
		directory->execute_cgiset     = false;
		directory->show_indexSet      = false;
		directory->follow_symlinksSet = false;
		directory->access_list        = NULL;
		directory->groups.number      = 0;
		directory->groups.array       = NULL;
		directory->passwordfile       = NULL;
		directory->groupfile          = NULL;
		init_charlist(&(directory->required_group));
		init_charlist(&(directory->image_referer));
		directory->imgref_replacement = NULL;
		directory->max_clients        = -1;
		directory->nr_of_clients      = 0;
		directory->upload_speed       = 0;
		directory->session_speed      = 0;
		directory->envir_str          = NULL;
		pthread_mutex_init(&(directory->client_mutex), NULL);

		directory->next               = NULL;
	}

	return directory;
}

static t_binding *new_binding(void) {
	t_binding *binding;

	if ((binding = (t_binding*)malloc(sizeof(t_binding))) != NULL) {
		binding->port                 = -1;
		binding->interface            = 0;
#ifdef HAVE_SSL
		binding->use_ssl              = false;
		binding->server_key           = NULL;
		binding->ssl_context          = NULL;
#endif
		binding->name                 = NULL;
		binding->socket               = -1;
		binding->enable_trace         = false;
		binding->max_keepalive        = 50;
		binding->max_request_size     = 65536;
		binding->time_for_1st_request = 5;
		binding->time_for_request     = 30;

		binding->next                 = NULL;
	}

	return binding;
}

t_config *default_config(void) {
	t_config *config;

	if ((config = (t_config*)malloc(sizeof(t_config))) != NULL) {
		config->throttle_config    = "throttle.conf";
		config->mimetype_config    = "mimetype.conf";

		config->server_root        = NULL;
		config->server_uid         = (uid_t)id_NOBODY;
		config->server_gid         = (gid_t)id_NOBODY;
		config->server_string      = "Hiawatha v"VERSION;
		config->groups.number      = 0;
		config->groups.array       = NULL;
#ifdef HAVE_SSL
		config->server_key         = CONFIG_DIR"/serverkey.pem";
#endif
		init_charlist(&(config->cgi_extension));
		config->total_connections  = 50;
		config->connections_per_ip = 10;
		config->time_for_cgi       = 5;
		config->first_host         = new_host();
#ifdef HAVE_PLUGIN
		config->plugin_uri         = NULL;
#endif
		config->mimetype           = NULL;
		config->directory          = NULL;
		config->throttle           = NULL;
		config->cgi_handler        = NULL;
		config->cgi_wrapper        = NULL;
		config->wrap_user_cgi      = false;

		config->system_logfile     = LOG_DIR"/system.log";
		config->garbage_logfile    = NULL;
		config->logfile_mask       = NULL;

		config->ban_on_garbage     = 0;
		config->ban_on_max_per_ip  = 2;
		config->ban_on_flooding    = 0;
		config->ban_on_max_request_size = 0;
		config->ban_on_cmdi        = 0;
		config->ban_on_sqli        = 0;
		config->ban_on_timeout     = 0;
		config->kick_on_ban        = false;
		config->reban_during_ban   = false;
		config->flooding_count     = 0;
		config->flooding_time      = 0;
		config->reconnect_delay    = 0;
		config->banlist_mask       = NULL;

#ifdef HAVE_COMMAND
		config->command_port       = NULL;
#endif
	}

	return config;
}

enum t_section new_section(char *line, enum t_section section, enum t_section_type *section_type) {
	if (strcmp(line, "}") == 0) {
		if (*section_type == bracket) {
			*section_type = end;
			return section;
		} else {
			return syntax_error;
		}
	} else if ((*line == '<') && (*(line + strlen(line) - 1) == '>')) {
		line++;
		if (*line == '/') {
			if (*section_type == tag) {
				line++;
				*section_type = end;
			} else {
				return syntax_error;
			}
		} else {
			*section_type = tag;
		}
	} else if (*(line + strlen(line) - 1) == '{') {
		*section_type = bracket;
	} else {
		return syntax_error;
	}

	*(line + strlen(line) - 1) = '\0';
	line = remove_spaces(line);

	if (strcmp(line, "binding") == 0) {
		return binding;
	} else if (strcmp(line, "virtualhost") == 0) {
		return virtualhost;
	} else if (strcmp(line, "directory") == 0) {
		return directory;
	}

	return syntax_error;
}

static bool valid_file(char *file) {
	bool retval = false;

	if (file != NULL) {
		if (strchr(file, '/') == NULL) {
			if (strlen(file) < 32) {
				retval = true;
			}
		}
	}

	return retval;
}

static bool valid_directory(char *dir) {
	bool retval = false;
	int len;

	if (dir != NULL) {
		if (*dir == '/') {
			if ((len = strlen(dir)) < 128) {
				if (*(dir + len - 1) != '/') {
					retval = true;
				}
			}
		}
	}

	return retval;
}

int check_configuration(t_config *config) {
#ifdef HAVE_SSL
	t_binding *binding;

	binding = config->binding;
	while (binding != NULL) {
		if (binding->use_ssl && (binding->server_key == NULL)) {
			binding->server_key = config->server_key;
		}
		binding = binding->next;
	}
#endif
	
	if (config->first_host->hostname.size == 0) {
		return mc_NO_HOSTNAME;
	}

	if (config->binding == NULL) {
		return mc_NO_BINDING;
	}

	if ((config->first_host->required_binding.size > 0)
#ifdef HAVE_SSL
		|| (config->first_host->require_ssl == true)
#endif
	) {
		return mc_REQUIRE;
	}

	return 0;
}

static int parse_yesno(char *yesno, bool *result) {
	if ((strcmp(yesno, "yes") == 0) || (strcmp(yesno, "true") == 0)) {
		*result = true;
	} else if ((strcmp(yesno, "no") == 0) || (strcmp(yesno, "false") == 0)) {
		*result = false;
	} else {
		return -1;
	}

	return 0;
}

int parse_passwordfile(char *line, char **pwdfile) {
	if (*pwdfile != NULL) {
		free(*pwdfile);
		*pwdfile = NULL;
	}
	if (strcmp(line, "none") != 0) {
		if (strlen(line) >= 256) {
			return -1;
		} else if ((*pwdfile = strdup(line)) == NULL) {
			return -1;
		}
	}

	return 0;
}
static bool system_setting(char *key, char *value, t_config *config) {
	bool retval = true;
	char *uid, *gid, *rest;
	t_cgi_handler *CGI;
#ifdef HAVE_COMMAND
	char *port, *password;
#endif

	do {
		if (strcmp(key, "banlistmask") == 0) {
			if ((config->banlist_mask = parse_accesslist(value, false, config->banlist_mask)) != NULL) {
				break;
			}
		} else if (strcmp(key, "banoncmdi") == 0) {
			if ((config->ban_on_cmdi = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonflooding") == 0) {
			if (split_string(value, &value, &rest, '/') == -1) {
			} else if ((config->flooding_count = str2int(value)) <= 0) {
			} else if (split_string(rest, &value, &rest, ':') != 0) {
			} else if ((config->flooding_time = str2int(value)) <= 0) {
			} else if ((config->ban_on_flooding = str2int(rest)) > 0) {
				config->flooding_time--;
				break;
			}
		} else if (strcmp(key, "banongarbage") == 0) {
			if ((config->ban_on_garbage = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonmaxperip") == 0) {
			if ((config->ban_on_max_per_ip = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonmaxreqsize") == 0) {
			if ((config->ban_on_max_request_size = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banonsqli") == 0) {
			if ((config->ban_on_sqli = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "banontimeout") == 0) {
			if ((config->ban_on_timeout = str2int(value)) != -1) {
				break;
			}
#ifdef HAVE_COMMAND
		} else if (strcmp(key, "commandchannel") == 0) {
			if (split_string(value, &port, &password, ',') == 0) {
				config->command_port = new_binding();
				config->command_port->interface = htonl(0x7F000001);
				if ((config->command_port->port = str2int(port)) > 0) {
					if ((config->command_port->name = strdup(password)) != NULL) {
						break;
					}
				} else {
					free(config->command_port);
				}
			}
#endif
		} else if (strcmp(key, "cgiextension") == 0) {
			if (parse_charlist(value, &(config->cgi_extension)) != -1) {
				break;
			}
		} else if (strcmp(key, "cgihandler") == 0) {
			if (split_string(value, &value, &rest, ':') == 0) {
				CGI = config->cgi_handler;
				config->cgi_handler = (t_cgi_handler*)malloc(sizeof(t_cgi_handler));
				config->cgi_handler->next = CGI;
				config->cgi_handler->handler = strdup(value);
				init_charlist(&(config->cgi_handler->extension));
				if (parse_charlist(rest, &(config->cgi_handler->extension)) != -1) {
					break;
				}
			}
		} else if (strcmp(key, "cgiwrapper") == 0) {
			if ((config->cgi_wrapper = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "connectionsperip") == 0) {
			if ((config->connections_per_ip = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "connectionstotal") == 0) {
			if ((config->total_connections = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "garbagelogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((config->garbage_logfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "kickonban") == 0) {
			if (parse_yesno(value, &(config->kick_on_ban)) == 0) {
				break;
			}
		} else if (strcmp(key, "rebanduringban") == 0) {
			if (parse_yesno(value, &(config->reban_during_ban)) == 0) {
				break;
			}
		} else if (strcmp(key, "logfilemask") == 0) {
			if ((config->logfile_mask = parse_accesslist(value, false, config->logfile_mask)) != NULL) {
				break;
			}
		} else if (strcmp(key, "mimetypeconfig") == 0) {
			if ((config->mimetype_config = strdup(value)) != NULL) {
				break;
			}
#ifdef HAVE_PLUGIN
		} else if (strcmp(key, "pluginuri") == 0) {
			if (strlen(value) < 16) {
				if ((config->plugin_uri = strdup(value)) != NULL) {
					break;
				}
			}
#endif
		} else if (strcmp(key, "reconnectdelay") == 0) {
			if ((config->reconnect_delay = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "serverid") == 0) {
			if (split_string(value, &uid, &gid, ':') == 0) {
				if ((config->server_uid = (uid_t)parse_userid(uid)) > (uid_t)0) {
					if ((config->server_gid = parse_groups(gid, &(config->groups))) > (gid_t)0) {
						break;
					}
				}
			} else if ((config->server_uid = (uid_t)parse_userid(value)) > (uid_t)0) {
				if ((config->server_gid = lookup_group_ids(config->server_uid, &(config->groups))) > (gid_t)0) {
					break;
				}
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "serverkey") == 0) {
			if ((config->server_key = strdup(value)) != NULL) {
				break;
			}
#endif
		} else if (strcmp(key, "serverroot") == 0) {
			if (valid_directory(value)) {
				if ((config->server_root = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "serverstring") == 0) {
			if (strlen(value) < 128) {
				if ((config->server_string = strdup(remove_spaces(value))) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "systemlogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((config->system_logfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "throttleconfig") == 0) {
			if ((config->throttle_config = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "timeforcgi") == 0) {
			if ((config->time_for_cgi = str2int(value)) >= 1) {
				break;
			}
		} else if (strcmp(key, "wrapusercgi") == 0) {
			if (parse_yesno(value, &(config->wrap_user_cgi)) == 0) {
				break;
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool user_setting(char *key, char *value, t_host *host) {
	bool retval = true;
	char *gid, *code;
	int result;

	do {
		if (strcmp(key, "accesslist") == 0) {
			if ((host->access_list = parse_accesslist(value, true, host->access_list)) != NULL) {
				break;
			}
		} else if (strcmp(key, "errorhandler") == 0) {
			result = split_string(value, &value, &code, ',');
			if ((strlen(value) < 128) && (*value == '/')) {
				if ((host->error_handler = strdup(value)) != NULL) {
					if (result == 0) {
						if ((host->return_code = str2int(code)) != -1) {
							break;
						}
					} else {
						break;
					}
				}
			}
		} else if (strcmp(key, "indexstyle") == 0) {
			if ((host->index_style = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "loginmessage") == 0) {
			if (strlen(value) < 64) {
				if ((host->login_message = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "passwordfile") == 0) {
			if (parse_passwordfile(value, &(host->passwordfile)) == 0) {
				break;
			}
		} else if (strcmp(key, "requiredgroup") == 0) {
			if (split_string(value, &gid, &value, ':') == 0) {
				if (strlen(gid) < 32) {
					if (parse_charlist(gid, &(host->required_group)) != -1) {
						if (parse_passwordfile(value, &(host->groupfile)) == 0) {
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "setenvir") == 0) {
			if ((host->envir_str = parse_keyvaluelist(value, host->envir_str, ':')) != NULL) {
				break;
			}
		} else if (strcmp(key, "showindex") == 0) {
			if (parse_yesno(value, &(host->show_index)) == 0) {
				break;
			}
		} else if (strcmp(key, "startfile") == 0) {
			if (valid_file(value)) {
				if ((host->start_file = strdup(value)) != NULL) {
					break;
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool host_setting(char *key, char *value, t_host *host) {
	bool retval = true;
	char *botname;
	t_denybotlist *denybot;
	t_keyvalue *alias;

	do {
		if (strcmp(key, "accesslogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((host->access_logfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "alias") == 0) {
			if ((host->alias = parse_keyvaluelist(value, host->alias, ':')) != NULL) {
				alias = host->alias;
				while (alias != NULL) {
					if ((valid_directory(alias->key) == false) || (valid_directory(alias->value) == false)) {
						retval = false;
						break;
					}
					alias = alias->next;
				}
				break;
			}
		} else if (strcmp(key, "cgiwrapid") == 0) {
			if ((host->cgi_wrap_id = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "denybot") == 0) {
			if (split_string(value, &botname, &value, ':') == 0) {
				denybot = (t_denybotlist*)malloc(sizeof(t_denybotlist));
				denybot->next = host->deny_bot;
				host->deny_bot = denybot;

				init_charlist(&(denybot->uri));
				denybot->bot = strdup(botname);
				if (parse_charlist(value, &(denybot->uri)) == 0) {
					break;
				}
			}
		} else if (strcmp(key, "errorlogfile") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				if ((host->error_logfile = strdup(value)) != NULL) {
					break;
				}
			}
		} else if (strcmp(key, "executecgi") == 0) {
			if (parse_yesno(value, &(host->execute_cgi)) == 0) {
				break;
			}
		} else if (strcmp(key, "followsymlinks") == 0) {
			if (parse_yesno(value, &(host->follow_symlinks)) == 0) {
				break;
			}
		} else if (strcmp(key, "hostname") == 0) {
			if (parse_charlist(value, &(host->hostname)) == 0) {
				break;
			}
		} else if (strcmp(key, "imagereferer") == 0) {
			if (split_string(value, &value, &(host->imgref_replacement), ':') == 0) {
				if ((host->imgref_replacement = strdup(host->imgref_replacement)) != NULL) {
					if (parse_charlist(value, &(host->image_referer)) == 0) {
						break;
					}
				}
			}
#ifdef HAVE_PLUGIN
		} else if (strcmp(key, "pluginactive") == 0) {
			if (parse_yesno(value, &(host->plugin_active)) == 0) {
				break;
			}
#endif
		} else if (strcmp(key, "preventcmdi") == 0) {
			if (parse_yesno(value, &(host->prevent_cmdi)) == 0) {
				break;
			}
		} else if (strcmp(key, "preventsqli") == 0) {
			if (parse_yesno(value, &(host->prevent_sqli)) == 0) {
				break;
			}
		} else if (strcmp(key, "preventxss") == 0) {
			if (parse_yesno(value, &(host->prevent_xss)) == 0) {
				break;
			}
		} else if (strcmp(key, "requiredbinding") == 0) {
			if (parse_charlist(value, &(host->required_binding)) == 0) {
				break;
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "requiressl") == 0) {
			if (parse_yesno(value, &(host->require_ssl)) == 0) {
				break;
			}
#endif
		} else if (strcmp(key, "userwebsites") == 0) {
			if (parse_yesno(value, &(host->user_websites)) == 0) {
				break;
			}
		} else if (strcmp(key, "volatileobject") == 0) {
			if ((strlen(value) < 256) && (*value == '/')) {
				host->volatile_object.size++;
				if ((host->volatile_object.item = (char**)realloc(host->volatile_object.item, host->volatile_object.size * sizeof(char*))) != NULL) {
					if ((*(host->volatile_object.item + host->volatile_object.size - 1) = strdup(value)) != NULL) {
						break;
					}
				}
			}
		} else if (strcmp(key, "websiteroot") == 0) {
			if (valid_directory(value)) {
				if ((host->website_root = strdup(value)) != NULL) {
					break;
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool directory_setting(char *key, char *value, t_directory *directory) {
	bool retval = true;
	char *gid, *maxclients;
	int length;

	do {
		if (strcmp(key, "accesslist") == 0) {
			if ((directory->access_list = parse_accesslist(value, true, directory->access_list)) != NULL) {
				break;
			}
		} else if (strcmp(key, "cgiwrapid") == 0) {
			if ((directory->cgi_wrap_id = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "executecgi") == 0) {
			if (parse_yesno(value, &(directory->execute_cgi)) == 0) {
				directory->execute_cgiset = true;
				break;
			}
		} else if (strcmp(key, "followsymlink") == 0) {
			if (parse_yesno(value, &(directory->follow_symlinks)) == 0) {
				directory->follow_symlinksSet = true;
				break;
			}
		} else if (strcmp(key, "imagereferer") == 0) {
			if (split_string(value, &value, &(directory->imgref_replacement), ':') == 0) {
				if ((directory->imgref_replacement = strdup(directory->imgref_replacement)) != NULL) {
					if (parse_charlist(value, &(directory->image_referer)) == 0) {
						break;
					}
				}
			}
		} else if (strcmp(key, "passwordfile") == 0) {
			if (parse_passwordfile(value, &(directory->passwordfile)) == 0) {
				break;
			}
		} else if (strcmp(key, "path") == 0) {
			length = strlen(value);
			if ((length < 128) && (*value == '/')) {
				if (*(value + length - 1) == '/') {
					if (length >= 3) {
						directory->path_match = part;
						if ((directory->path = strdup(value)) != NULL) {
							break;
						}
					}
				} else {
					if (length >= 2) {
						directory->path_match = root;
						if ((directory->path = (char*)malloc(length + 2)) != NULL) {
							memcpy(directory->path, value, length);
							memcpy(directory->path + length, "/\0", 2);
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "requiredgroup") == 0) {
			if (split_string(value, &gid, &value, ':') == 0) {
				if (strlen(gid) < 32) {
					if (parse_charlist(gid, &(directory->required_group)) != -1) {
						if (parse_passwordfile(value, &(directory->groupfile)) == 0) {
							break;
						}
					}
				}
			}
		} else if (strcmp(key, "setenvir") == 0) {
			if ((directory->envir_str = parse_keyvaluelist(value, directory->envir_str, 'c')) != NULL) {
				break;
			}
		} else if (strcmp(key, "showindex") == 0) {
			if (parse_yesno(value, &(directory->show_index)) == 0) {
				directory->show_indexSet = true;
				break;
			}
		} else if (strcmp(key, "uploadspeed") == 0) {
			if (split_string(value, &value, &maxclients, ',') == 0) {
				if ((directory->upload_speed = str2int(value)) > 0) {
					directory->upload_speed <<= 10;
					if ((directory->max_clients = str2int(maxclients)) > 0) {
						break;
					}
				}
			}
		}
		retval = false;
	} while (false);

	return retval;
}

static bool binding_setting(char *key, char *value, t_binding *binding) {
	bool retval = true;
	char *rest;

	do {
		if (strcmp(key, "enabletrace") == 0) {
			if (parse_yesno(value, &(binding->enable_trace)) == 0) {
				break;
			}
		} else if (strcmp(key, "interface") == 0) {
			if (parse_ip(value, &(binding->interface)) == 0) {
				break;
			}
		} else if (strcmp(key, "maxkeepalive") == 0) {
			if ((binding->max_keepalive = str2int(value)) != -1) {
				break;
			}
		} else if (strcmp(key, "maxrequestsize") == 0) {
			if ((binding->max_request_size = str2int(value)) > 0) {
				binding->max_request_size <<= 10;
				break;
			}
		} else if (strcmp(key, "name") == 0) {
			if ((binding->name = strdup(value)) != NULL) {
				break;
			}
		} else if (strcmp(key, "port") == 0) {
			if ((binding->port = str2int(value)) > 0) {
				if (binding->port < 65536) {
					break;
				}
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "serverkey") == 0) {
			if ((binding->server_key = strdup(value)) != NULL) {
				binding->use_ssl = true;
				break;
			}
#endif
		} else if (strcmp(key, "timeforrequest") == 0) {
			if (split_string(value, &value, &rest, ',') == 0) {
				if ((binding->time_for_1st_request = str2int(value)) >= 1) {
					if ((binding->time_for_request = str2int(rest)) >= 1) {
						break;
					}
				}
			} else if ((binding->time_for_request = str2int(value)) >= 1) {
				binding->time_for_1st_request = binding->time_for_request;
				break;
			}
#ifdef HAVE_SSL
		} else if (strcmp(key, "usessl") == 0) {
			if (parse_yesno(value, &(binding->use_ssl)) == 0) {
				break;
			}
#endif
		}
		retval = false;
	} while (false);

	return retval;
}

void print_configerror(char *configfile, int code) {
	if (code != mc_INCLUDE) {
		printf("\n[%s] Fatal error: ", configfile);
		switch (code) {
			case mc_READ_ERROR:
				printf("can't read configfile %s.\n", configfile);
				break;
			case mc_DIR_CONFLICT:
				printf("server_root conflicts with user_websites.\n");
				break;
			case mc_NO_HOSTNAME:
				printf("hostname missing.\n");
				break;
			case mc_NO_BINDING:
				printf("No bindings present.\n");
				break;
			case mc_REQUIRE:
				printf(
#ifdef HAVE_SSL
					"require_ssl and "
#endif
					"required_binding not allowed outside VirutalHost block.\n");
				break;
			default:
				printf("invalid option in line %d.\n", code);
		}   
	}
}

bool replace_variables(char **line) {
	bool replaced = false;
	t_keyvalue *variable;
	char *new;

	variable = variables;
	while (variable != NULL) {
		if ((new = str_replace(*line, variable->key, variable->value)) != NULL) {
			if (replaced) {
				free(*line);
			}
			*line = new;
			replaced = true;
		}
		variable = variable->next;
	}

	return replaced;
}

int read_main_configfile(char *file, t_config *config) {
	int  retval = 0, counter = 0;
	FILE *fp;
	char *line, *key, *value;
	bool variables_replaced;
	enum t_section section = none;
	enum t_section_type section_type = end;
	t_host *current_host;
	t_directory *current_directory = NULL;
	t_binding *current_binding = NULL;

	/* Read and parse Hiawatha configurationfile.
	 */
	if ((fp = fopen(file, "r")) != NULL) {
		current_host = config->first_host;
		line = (char*)malloc(257);
		*(line + 256) = '\0';
		while (fgets(line, 256, fp) != NULL) {
			counter++;
			key = strip_line(line);
			if (*key != '\0') {
				variables_replaced = false;
				if (strncmp(key, "set ", 4) == 0) {
					if ((variables = parse_keyvaluelist(key + 4, variables, '=')) == NULL) {
						retval = counter;
					}
				} else if (split_string(key, &key, &value, '=') != -1) {
					if (strlen(value) > 0) {
						strlower(key);
						variables_replaced = replace_variables(&value);
						do {
							if (section == none) {
								if (system_setting(key, value, config)) {
									break;
								}
							}
							if ((section == none) || (section == virtualhost)) {
								if (host_setting(key, value, current_host)) {
									break;
								} else if (user_setting(key, value, current_host)) {
									break;
								}
							} else if (section == directory) {
								if (directory_setting(key, value, current_directory)) {
									break;
								}
							} else if (section == binding) {
								if (binding_setting(key, value, current_binding)) {
									break;
								}
							}
							retval = counter;
						} while (false);
					} else {
						retval = counter;
					}
				} else {
					key = strlower(key);
					if (strncmp(key, "include ", 8) == 0) {
						value = key + 8;
						variables_replaced = replace_variables(&value);
						if (section == none) {
							if (including == false) {
								including = true;
								if ((retval = read_main_configfile(value, config)) != 0) {
									print_configerror(value, retval);
									retval = mc_INCLUDE;
								}
								including = false;
							} else {
								retval = counter;
							}
						} else {
							retval = counter;
						}
					} else switch ((section = new_section(key, section, &section_type))) {
						case binding:
							// binding section
							if (section_type != end) {
								if (config->binding != NULL) {
									current_binding = config->binding;
									while (current_binding->next != NULL) {
										current_binding = current_binding->next;
									}
									current_binding->next = new_binding();
									current_binding = current_binding->next;
								} else {
									config->binding = new_binding();
									current_binding = config->binding;
								}
							} else {
								if (current_binding->port != -1) {
									current_binding = NULL;
									section = none;
								} else {
									retval = counter;
								}
							}
							break;
						case virtualhost:
							// VirtualHost section
							if (section_type != end) {
								while (current_host->next != NULL) {
									current_host = current_host->next;
								}
								current_host->next = new_host();
								current_host = current_host->next;
							} else {
								if (current_host->hostname.size > 0) {
									current_host = config->first_host;
									section = none;
								} else {
									retval = counter;
								}
							}
							break;
						case directory:
							// directory section
							if (section_type != end) {
								if (config->directory != NULL) {
									current_directory = config->directory;
									while (current_directory->next != NULL) {
										current_directory = current_directory->next;
									}
									current_directory->next = new_directory();
									current_directory = current_directory->next;
								} else {
									config->directory = new_directory();
									current_directory = config->directory;
								}
							} else {
								if (config->directory->path != NULL) {
									current_directory = NULL;
									section = none;
								} else {
									retval = counter;
								}
							}
							break;
						default:
							retval = counter;
					}
				}
				if (variables_replaced) {
					free(value);
				}
			}
			if (retval != 0) {
				break;
			}
		}
		fclose(fp);
		free(line);
		if (including == false) {
			remove_keyvaluelist(variables);
			variables = NULL;
		}

		if ((retval == 0) && (section != none)) {
			retval = counter;
		}
	} else {
		retval = mc_READ_ERROR;
	}

	return retval;
}

int read_user_configfile(char *file, t_host *host) {
	int  retval = 0;
	FILE *fp;
	char *line, *key, *value;

	if ((fp = fopen(file, "r")) != NULL) {
		if ((line = (char*)malloc(257)) != NULL) {
			*(line + 256) = '\0';
			while (fgets(line, 256, fp) != NULL) {
				key = strip_line(line);
				if (*key != '\0') {
					if (split_string(key, &key, &value, '=') != -1) {
						strlower(key);
						user_setting(key, value, host);
					}
				}
			}
			free(line);
		} else {
			retval = -1;
		}
		fclose(fp);
	}

	return retval;
}

t_host *get_hostrecord(t_host *host, char *hostname, t_binding *binding) {
	char *dots;
	int i, len1, len2;

	if (hostname != NULL) {
		if ((dots = strchr(hostname, ':')) != NULL) {
			*dots = '\0';
		}

		len1 = strlen(hostname);
		while (host != NULL) {
			for (i = 0; i < host->hostname.size; i++) {
				if (host->required_binding.size != 0) {
					if (in_charlist(binding->name, &(host->required_binding)) == false) {
						// Binding not allowed
						continue;
					}
				}
				if (strcmp(hostname, *(host->hostname.item + i)) == 0) {
					// Exact match
					return host;
				} else if (strncmp(*(host->hostname.item + i), "*.", 2) == 0) {
					// Wildcard in configuration
					if (strcmp(hostname, *(host->hostname.item + i) + 2) == 0) {
						// Only domainname requested
						return host;
					} else {
						len2 = strlen(*(host->hostname.item + i));
						if (len1 >= len2) {
							if (strcmp(hostname + len1 - len2 + 1, *(host->hostname.item + i) + 1) == 0) {
								// Wildcard match for hostname
								return host;
							}
						}
					}
				}
			}

			host = host->next;
		}
	}

	return NULL;
}

int read_throttleconfig(char *configfile, t_throttle **throttle) {
	FILE *fp;
	char *line, *key, *value;
	t_throttle *throt = NULL;
	int speed;

	*throttle = NULL;
	if ((fp = fopen(configfile, "r")) == NULL) {
		return -1;
	}

	line = (char*)malloc(257);
	*(line + 256) = '\0';
	while (fgets(line, 256, fp) != NULL) {
		if ((*line != '#') && (strlen(line) > 0)) {
			if (split_string(line, &key, &value, ':') != -1) {
				if (((*key == '.') || (strchr(key, '/') != NULL)) && (speed = str2int(value)) > 0) {
					if (throt == NULL) {
						*throttle = throt = (t_throttle*)malloc(sizeof(t_throttle));
					} else {
						throt->next = (t_throttle*)malloc(sizeof(t_throttle));
						throt = throt->next;
					}
					throt->next = NULL;
					throt->filetype = strlower(strdup(key));
					throt->upload_speed = speed << 10; // convert to kilobytes
				}
			}
		}
	}
	fclose(fp);
	free(line);

	return 0;
}

unsigned short get_throttlespeed(char *type, t_throttle *throttle) {
	t_throttle *throt;
	unsigned long speed = 0;
	int len_type, len_throt;
	char *type_lower;

	if (type != NULL) {
		type_lower = strlower(strdup(type));
		len_type = strlen(type);

		throt = throttle;
		while (throt != NULL) {
			len_throt = strlen(throt->filetype);
			if (len_type >= len_throt) {
				if (memcmp(throt->filetype, type_lower, len_throt) == 0) {
					speed = throt->upload_speed;
					break;
				}
			}
			throt = throt->next;
		}
		free(type_lower);
	}

	return speed;
}
