/* Hiawatha | session.c
 *
 * All the routines to handle a session. A session is a connection from a
 * client to the webserver.
 */
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <pwd.h>
#include "libstr.h"
#include "session.h"
#include "liblist.h"
#ifdef HAVE_SSL
#include "libssl.h"
#endif

int new_client_id = 0;

#ifdef HAVE_COMMAND
extern unsigned long clients_served;
#endif

char *reasons_for_403[5] = {
	":filesystem",
	":access_list",
	":noCGIexec",
	":hasSymlink",
	":deny_bot"
};

/* Set the entries in a session-record to the default values.
 */
static void clear_session(t_session *session) {
	session->head_request = false;
	session->is_cgi_script = false;
	session->cgi_handler = NULL;
	session->method = NULL;
	session->uri = NULL;
	session->uri_is_dir = false;
	session->accept_gzip = false;
	session->encode_gzip = false;
	session->vars = NULL;
	session->vars_copied = false;
	session->http_version = NULL;
	session->headerfields = NULL;
	session->body = NULL;
	session->local_user = NULL;
#ifdef HAVE_SSL
	session->cause_of_301 = missing_slash;
#endif
	session->body_copied = false;
	session->header_length = 0;
	session->content_length = 0;
	session->file_on_disk = NULL;
	session->mimetype = NULL;
	session->host = session->config->first_host;
	session->host_copied = false;
	session->throttle = 0;
	session->throttle_timer = 0;
	session->bytecounter = 0;
	session->part_of_dirspeed = false;
	session->remote_user = NULL;
	session->directory = NULL;
	session->handling_error = false;
	session->reason_for_403 = "";
	session->cookie = NULL;
	session->cookie_copied = false;
	session->websiteroot_copied = false;
	session->bytes_sent = 0;
	session->output_size = 0;
	session->return_code = 200;
	session->error_code = -1;
}

/* Initialize a session-record.
 */
void init_session(t_session *session) {
#ifdef HAVE_COMMAND
	clients_served++;
#endif
	if ((session->client_id = new_client_id++) == MAX_CHILD_ID) {
		new_client_id = 0;
	}
	session->force_quit = false;
	session->kept_alive = 0;
	session->flooding_counter = 0;
	session->flooding_timer = session->config->flooding_time;
#ifdef HAVE_SSL
	session->ssl_data = NULL;
#endif

	session->request = NULL;
	session->buffer_size = 0;
	session->bytes_in_buffer = 0;

	clear_session(session);
}

/* Reset a session-record for reuse.
 */
void reset_session(t_session *session) {
	int size;

	sfree(session->local_user);
	sfree(session->remote_user);
	sfree(session->file_on_disk);
	//sfree(session->chunk_buffer);
	remove_headerfields(session);
	if (session->directory != NULL) {
		pthread_mutex_lock(&(session->directory->client_mutex));
		if (session->part_of_dirspeed) {
			if (--session->directory->nr_of_clients == 0) {
				session->directory->session_speed = session->directory->upload_speed;
			} else {
				session->directory->session_speed = session->directory->upload_speed / session->directory->nr_of_clients;
			}
		}
		pthread_mutex_unlock(&(session->directory->client_mutex));
	}
	if (session->vars_copied) {
		free(session->vars);
	}
	if (session->body_copied) {
		free(session->body);
	}
	if (session->cookie_copied) {
		free(session->cookie);
	}

	/* HTTP pipelining
	 */
	size = session->header_length + session->content_length;
	if ((session->bytes_in_buffer > size) && session->keep_alive) {
		session->bytes_in_buffer -= size;
		memcpy(session->request, session->request + size, session->bytes_in_buffer);
		*(session->request + session->bytes_in_buffer) = '\0';
	} else {
		sfree(session->request);
		session->request = NULL;
		session->buffer_size = 0;
		session->bytes_in_buffer = 0;
	}
	
	clear_session(session);
	session->flooding_counter++;
}

/* Parse the HTTP headerfields from the received request.
 */
t_headerfield *parse_headerfields(char *line) {
	t_headerfield *first, *headerfield;
	char *value;

	if ((first = headerfield = (t_headerfield*)malloc(sizeof(t_headerfield))) == NULL) {
		return NULL;
	}
	headerfield->data = line;
	headerfield->next = NULL;
	while (*line != '\0') {
		if (*line == '\r') {
			*line = '\0';
			if ((*(line + 1) == '\n') && (*(line + 2) != '\r') && (*(line + 2) != '\0')) {
				if ((headerfield->next = (t_headerfield*)malloc(sizeof(t_headerfield))) == NULL) {
					return first;
				}
				headerfield = headerfield->next;
				headerfield->next = NULL;
				headerfield->data = line + 2;
			}
		}
		line++;
	}

	headerfield = first;
	while (headerfield != NULL) {
		if ((value = strchr(headerfield->data, ':')) != NULL) {
			do {
				value++;
			} while ((*value == ' ') && (*value != '\0'));
			headerfield->value_offset = (value - headerfield->data);
		} else {
			headerfield->value_offset = 0;
		}
		headerfield = headerfield->next;
	}

	return first;
}

/* Set the reason for the 403 HTTP error
 */
void set_403_reason(t_session *session, int reason) {
	session->reason_for_403 = reasons_for_403[reason];
}

/* Search for a headerfield and return its value.
 */
char *get_headerfield(char *key, t_headerfield *headerfields) {
	char *retval = NULL;
	int len;

	len = strlen(key);
	while (headerfields != NULL) {
		if (str_ncompare(headerfields->data, key, len) == 0) {
			retval = headerfields->data + headerfields->value_offset;
			break;
		}
		headerfields = headerfields->next;
	}

	return retval;
}

/* free() a list of headerfields.
 */
void remove_headerfields(t_session *session) {
	t_headerfield *to_be_removed;

	while (session->headerfields != NULL) {
		to_be_removed = session->headerfields;
		session->headerfields = session->headerfields->next;

		free(to_be_removed);
	}
}

/* Return the path of the user's homedirectory.
 */
int get_homedir(t_session *session, char *username) {
	struct passwd *pwd;

	if ((pwd = getpwnam(username)) != NULL) {
		session->host->website_root = (char*)malloc(strlen(pwd->pw_dir) + 13);
		strcpy(session->host->website_root, pwd->pw_dir);
		strcat(session->host->website_root, "/public_html");
		session->websiteroot_copied = true;
		return 200;
	}

	return 404;
}

/* Dupliacte the active host-record. The duplicate can now savely be altered
 * and will be used during the session.
 */
bool duplicate_host(t_session *session) {
	t_host *new_host;

	if ((session->host != NULL) && (session->host_copied == false)) {
		if ((new_host = (t_host*)malloc(sizeof(t_host))) != NULL) {
			memcpy(new_host, session->host, sizeof(t_host));
			new_host->next = NULL;
			session->host = new_host;
			session->host_copied = true;
		} else {
			return false;
		}
	}

	return true;
}

/* Restore the active host-record.
 */
void restore_host(t_session *session) {
	if (session->host_copied) {
		if (session->websiteroot_copied) {
			free(session->host->website_root);
			session->websiteroot_copied = false;
		}

		free(session->host);
		session->host_copied = false;
	}
	session->host = session->config->first_host;
}

/* Is the requested file marked as volatile?
 */
bool is_volatile_object(t_session *session) {
	bool retval = false;
	int i;

	for (i = 0; i < session->host->volatile_object.size; i++) {
		if (strcmp(session->file_on_disk, *(session->host->volatile_object.item + i)) == 0) {
			retval = true;
			break;
		}
	}

	return retval;
}

/* Copy the settings from a directory-record to the active host-record.
 */
int copy_directory_settings(t_session *session) {
	size_t path_length;
	t_directory *dir;
	bool match;

	dir = session->config->directory;
	while (dir != NULL) {
		path_length = strlen(dir->path);
		if (strlen(session->file_on_disk) >= path_length) {
			if (dir->path_match == root) {
				match = (strncmp(session->file_on_disk, dir->path, path_length) == 0);
			} else {
				match = (strstr(session->file_on_disk, dir->path) != NULL);
			}
			if (match) {
				if (duplicate_host(session)) {
					session->directory = dir;

					if (dir->max_clients > -1) {
						pthread_mutex_lock(&(dir->client_mutex));
						if (dir->nr_of_clients < dir->max_clients) {
							session->throttle = dir->session_speed = dir->upload_speed / ++dir->nr_of_clients;
							pthread_mutex_unlock(&(dir->client_mutex));
							session->part_of_dirspeed = true;
						} else {
							pthread_mutex_unlock(&(dir->client_mutex));
							return 503;
						}
					}
					if (dir->cgi_wrap_id != NULL) {
						session->host->cgi_wrap_id = dir->cgi_wrap_id;
					}
					if (dir->execute_cgiset) {
						session->host->execute_cgi = dir->execute_cgi;
					}
					if (dir->show_indexSet) {
						session->host->show_index = dir->show_index;
					}
					if (dir->follow_symlinksSet) {
						session->host->follow_symlinks = dir->follow_symlinks;
					}
					if (dir->access_list != NULL) {
						session->host->access_list = dir->access_list;
					}
					if (dir->image_referer.size > 0) {
						session->host->image_referer.size = dir->image_referer.size;
						session->host->image_referer.item = dir->image_referer.item;
						session->host->imgref_replacement = dir->imgref_replacement;
					}
					if (dir->passwordfile != NULL) {
						session->host->passwordfile = dir->passwordfile;
					}
					if (dir->groupfile != NULL) {
						session->host->groupfile = dir->groupfile;
						session->host->required_group = dir->required_group;
					}
					break;
				} else {
					return 500;
				}
			}
		}
		dir = dir->next;
	}

	return 200;
}

/* Wrapper for setenv
 */
static int add_to_environment(char *key, char *value, char *buffer, int pos, int max) {
	int key_len, value_len;

	if (buffer == NULL) {
		setenv(key, value, 1);

		return 0;
	} else {
		key_len = strlen(key);
		value_len = strlen(value);

		if (pos + 2 + key_len + value_len <= max) {
			buffer += pos;
			*buffer = (char)key_len;
			*(buffer + 1) = (char)value_len;
			memcpy(buffer + 2, key, key_len);
			memcpy(buffer + 2 + key_len, value, value_len);

			return 2 + key_len + value_len;
		} else {
			return 0;
		}
	}
}

/* Copy a headerfield to an environment setting
 */
static int headerfield_to_environment(t_session *session, char *key, char *envir, char *buffer, int pos, int max) {
	char *value;

	if ((value = get_headerfield(key, session->headerfields)) != NULL) {
		return add_to_environment(envir, value, buffer, pos, max);
	} else {
		return 0;
	}
}

/* Set environment variables for CGI script.
 */
int set_environment(t_session *session, char *buffer, int max) {
	char ip[16], len[16], value[16];
	unsigned char *ip_address;
	t_keyvalue *envir;
	int pos = 0;

	pos += add_to_environment("GATEWAY_INTERFACE", "CGI/1.1", buffer, pos, max);
	pos += add_to_environment("DOCUMENT_ROOT", session->host->website_root, buffer, pos, max);
	pos += add_to_environment("REQUEST_METHOD", session->method, buffer, pos, max);
	pos += add_to_environment("REQUEST_URI", session->uri, buffer, pos, max);
	pos += add_to_environment("SCRIPT_NAME", session->uri, buffer, pos, max);
	pos += add_to_environment("SCRIPT_FILENAME", session->file_on_disk, buffer, pos, max);

	ip_address = (unsigned char*)&(session->ip_address);
	sprintf(ip, "%hhu.%hhu.%hhu.%hhu", ip_address[0], ip_address[1], ip_address[2], ip_address[3]);
	pos += add_to_environment("REMOTE_ADDR", ip, buffer, pos, max);

	if (session->vars != NULL) {
		pos += add_to_environment("QUERY_STRING", session->vars, buffer, pos, max);
	}

	if (session->body != NULL) {
		sprintf(len, "%ld", session->content_length);
		pos += add_to_environment("CONTENT_LENGTH", len, buffer, pos, max);
		pos + headerfield_to_environment(session, "Content-Type:", "CONTENT_TYPE", buffer, pos, max);
	}

	sprintf(value, "%d", session->binding->port);
	pos += add_to_environment("SERVER_PORT", value, buffer, pos, max);
	pos += add_to_environment("SERVER_NAME", *(session->host->hostname.item), buffer, pos, max);
	pos += add_to_environment("SERVER_PROTOCOL", session->http_version, buffer, pos, max);
	pos += add_to_environment("SERVER_SOFTWARE", "Hiawatha v"VERSION, buffer, pos, max);
	pos += add_to_environment("REDIRECT_STATUS", "CGI", buffer, pos, max);

	if (session->remote_user != NULL) {
		pos += add_to_environment("AUTH_TYPE", "Basic", buffer, pos, max);
		pos += add_to_environment("REMOTE_USER", session->remote_user, buffer, pos, max);
	}

#ifdef HAVE_SSL
	if (session->binding->use_ssl) {
		pos += add_to_environment("CONNECTION_TYPE", "SSL", buffer, pos, max);
	} else
#endif
		pos += add_to_environment("CONNECTION_TYPE", "plain", buffer, pos, max);

	pos += headerfield_to_environment(session, "Accept:", "HTTP_ACCEPT", buffer, pos, max);
	pos += headerfield_to_environment(session, "Accept-Charset:", "HTTP_ACCEPT_CHARSET", buffer, pos, max);
	pos += headerfield_to_environment(session, "Accept-Language:", "HTTP_ACCEPT_LANGUAGE", buffer, pos, max);
	pos += headerfield_to_environment(session, "Client-IP:", "HTTP_CLIENT_IP", buffer, pos, max);
	pos += headerfield_to_environment(session, "From:", "HTTP_FROM", buffer, pos, max);
	pos += headerfield_to_environment(session, "Host:", "HTTP_HOST", buffer, pos, max);
	pos += headerfield_to_environment(session, "Range:", "HTTP_RANGE", buffer, pos, max);
	pos += headerfield_to_environment(session, "Referer:", "HTTP_REFERER", buffer, pos, max);
	pos += headerfield_to_environment(session, "User-Agent:", "HTTP_USER_AGENT", buffer, pos, max);
	pos += headerfield_to_environment(session, "Via:", "HTTP_VIA", buffer, pos, max);
	pos += headerfield_to_environment(session, "X-Forwarded-For:", "HTTP_X_FORWARDED_FOR", buffer, pos, max);

	sprintf(value, "%d", session->return_code);
	pos += add_to_environment("HTTP_RETURN_CODE", value, buffer, pos, max);
	if (session->error_code != -1) {
		sprintf(value, "%d", session->error_code);
		pos += add_to_environment("HTTP_GENERATED_ERROR", value, buffer, pos, max);
	}

	if (session->cookie != NULL) {
		pos += add_to_environment("HTTP_COOKIE", session->cookie, buffer, pos, max);
	} else {
		pos += headerfield_to_environment(session, "Cookie:", "HTTP_COOKIE", buffer, pos, max);
	}

	envir = session->host->envir_str;
	while (envir != NULL) {
		pos += add_to_environment(envir->key, envir->value, buffer, pos, max);
		envir = envir->next;
	}

	return pos;
}

/* Complete path in chrooted session?
 *  *  */
char *file_in_chroot(t_session *session, char *file) {
	size_t length;

	if (session->config->server_root != NULL) {
		length = strlen(session->config->server_root);
		if (strlen(session->host->passwordfile) > length) {
			if (memcmp(session->host->passwordfile, session->config->server_root, length) == 0) {
				file += length;
			}
		}
	}

	return file;
}

/* Check if User-Agent string contains deny_bot substring.
 */
bool client_is_rejected_bot(t_session *session) {
	int i, urilen, len;
	char *useragent;
	t_denybotlist *botlist;

	if (session->host->deny_bot == NULL) {
		return false;
	} else if ((useragent = get_headerfield("User-Agent:", session->headerfields)) == NULL) {
		return false;
	}

	urilen = strlen(session->uri);
	botlist = session->host->deny_bot;
	while (botlist != NULL) {
		if (strstr(useragent, botlist->bot) != NULL) {
			for (i = 0; i < botlist->uri.size; i++) {
				len = strlen(*(botlist->uri.item + i));
				if (urilen >= len) {
					if (memcmp(*(botlist->uri.item + i), session->uri, len) == 0) {
						return true;
					}
				}
			}
		}
		botlist = botlist->next;
	}

	return false;
}

void close_socket(t_session *session) {
	if (session->socket_open) {
#ifdef HAVE_SSL
		if (session->binding->use_ssl) {
			ssl_close(session->ssl_data);
		}
#endif
		close(session->client_socket);
		session->socket_open = false;
	}
}
