#include "config.h"

#ifdef HAVE_TOOLKIT

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "toolkit.h"
#include "libstr.h"
#include "libfs.h"
#include "alternative.h"

#define MAX_RULE_LOOP   20
#define MAX_MATCH_LOOP  20

t_url_toolkit *find_toolkit(char *toolkit_id, t_url_toolkit *url_toolkit) {
	while (url_toolkit != NULL) {
		if (strcmp(url_toolkit->toolkit_id, toolkit_id) == 0) {
			return url_toolkit;
		}
		url_toolkit = url_toolkit->next;
	}

	return NULL;
}

static int replace(char *src, int ofs, int len, char *rep, char **dst) {
	int len_rep;

	len_rep = strlen(rep);
	if ((*dst = (char*)malloc(strlen(src) - len + len_rep + 1)) == NULL) {
		return -1;
	}

	memcpy(*dst, src, ofs);
	memcpy(*dst + ofs, rep, len_rep);
	strcpy(*dst + ofs + len_rep, src + ofs + len);

	return 0;
}

bool toolkit_setting(char *key, char *value, t_url_toolkit *toolkit) {
	t_toolkit_rule *new_rule, *rule;
	char *rest;
	int loop;

	if (strcmp(key, "toolkitid") == 0) {
		if ((toolkit->toolkit_id = strdup(value)) != NULL) {
			return true;
		}
	} else {
		if ((new_rule = (t_toolkit_rule*)malloc(sizeof(t_toolkit_rule))) == NULL) {
			return false;
		} else if (toolkit->toolkit_rule == NULL) {
			toolkit->toolkit_rule = new_rule;
		} else {
			rule = toolkit->toolkit_rule;
			while (rule->next != NULL) {
				rule = rule->next;
			}
			rule->next = new_rule;
		}
		new_rule->parameter = NULL;
		new_rule->done = td_continue;
		new_rule->match_loop = 1;
		new_rule->next = NULL;

		if (strcmp(key, "match") == 0) {
			/* Match
			 */
			if (split_string(value, &value, &rest, ' ') == -1) {
				return false;
			} else if (regcomp(&(new_rule->pattern), value, REG_EXTENDED) != 0) {
				return false;
			}
			split_string(rest, &value, &rest, ' ');
			
			if (strcasecmp(value, "rewrite") == 0) {
				/* Match Rewrite
				 */
				new_rule->action = ta_match_rewrite;
				new_rule->done = td_exit;

				split_string(rest, &value, &rest, ' ');
				if (strlen(value) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(value)) == NULL) {
					return false;
				}

				if (rest != NULL) {
					split_string(rest, &value, &rest, ' ');
					if ((loop = str2int(value)) > 0) {
						if (loop > MAX_MATCH_LOOP) {
							return false;
						} else if ((value = rest) == NULL) {
							return false;
						}
						new_rule->match_loop = loop;
					} else if (rest != NULL) {
						return false;
					}

					if (strcasecmp(value, "continue") == 0) {
						new_rule->done = td_continue;
					} else if (strcasecmp(value, "return") == 0) {
						new_rule->done = td_return;
					} else {
						return false;
					}
				}
			} else if (strcasecmp(value, "redirect") == 0) {
				/* Match Redirect
				 */
				new_rule->action = ta_match_redirect;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "goto") == 0) {
				/* Match Goto
				 */
				new_rule->action = ta_match_goto;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "call") == 0) {
				/* Match Call
				 */
				new_rule->action = ta_match_call;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "return") == 0) {
				/* Match Return
				 */
				new_rule->action = ta_match_return;
			} else if (strcasecmp(value, "exit") == 0) {
				/* Match Exit
				 */
				new_rule->action = ta_match_exit;
			} else if (strcasecmp(value, "denyaccess") == 0) {
				/* Match DenyAccess
				 */
				new_rule->action = ta_match_deny_access;
			} else if (strcasecmp(value, "skip") == 0) {
				/* Match Skip
				 */
				new_rule->action = ta_match_skip;
				if ((new_rule->value = str2int(rest)) < 1) {
					return false;
				}
			} else if (strcasecmp(value, "usefastcgi") == 0) {
				/* Match UseFastCGI
				 */
				new_rule->action = ta_match_fastcgi;
				new_rule->parameter = strdup(rest);
			} else {
				return false;
			}
		} else if (strcmp(key, "call") == 0) {
			/* Call
			 */
			new_rule->action = ta_call;
			if (strlen(value) == 0) {
				return false;
			} else if ((new_rule->parameter = strdup(value)) == NULL) {
				return false;
			}
		} else if (strcmp(key, "skip") == 0) {
			/* Skip
			 */
			new_rule->action = ta_skip;
			if ((new_rule->value = str2int(value)) < 1) {
				return false;
			}
		} else if (strcmp(key, "requesturi") == 0) {
			/* RequestURI
			 */
			new_rule->action = ta_requesturi;
			if (split_string(value, &value, &rest, ' ') == -1) {
				return false;
			}
			if (strcasecmp(value, "exists") == 0) {
				new_rule->value = IU_EXISTS;
			} else if (strcasecmp(value, "isfile") == 0) {
				new_rule->value = IU_ISFILE;
			} else if (strcasecmp(value, "isdir") == 0) {
				new_rule->value = IU_ISDIR;
			} else {
				return false;
			}
			if (strcasecmp(rest, "return") == 0) {
				new_rule->done = td_return;
			} else if (strcasecmp(rest, "exit") == 0) {
				new_rule->done = td_exit;
			} else {
				return false;
			}
		} else {
			return false;
		}

		return true;
	}

	return false;
}

bool toolkit_rules_oke(t_url_toolkit *url_toolkit) {
	t_url_toolkit *toolkit;
	t_toolkit_rule *rule;

	toolkit = url_toolkit;
	while (toolkit != NULL) {
		if (toolkit->toolkit_id == NULL) {
			fprintf(stderr, "A ToolkitID is missing in an UrlToolkit section.\n");
			return false;
		}

		rule = toolkit->toolkit_rule;
		while (rule != NULL) {
			if ((rule->action == ta_match_goto) || (rule->action == ta_match_call) || (rule->action == ta_call)) {
				if (rule->parameter == NULL) {
					fprintf(stderr, "Missing parameter in toolkit rule '%s'.\n", toolkit->toolkit_id);
					return false;
				} else if (find_toolkit(rule->parameter, url_toolkit) == NULL) {
					fprintf(stderr, "Unknown ToolkitID in Goto/Call in toolkit rule '%s'.\n", toolkit->toolkit_id);
					return false;
				}
			}
			rule = rule->next;
		}
		toolkit = toolkit->next;
	}

	return true;
}

static int do_rewrite(char *url, regex_t *regexp, char *rep, char **new_url, int loop) {
	int ofs, len, len_rep, i, n;
	regmatch_t pmatch[10];
	char *repl, *c, *sub, *tmp;
	bool replaced;

	*new_url = NULL;
	while (loop-- > 0) {
		if (regexec(regexp, url, 10, pmatch, 0) == REG_NOMATCH) {
			break;
		}
		if ((ofs = pmatch[0].rm_so) == -1) {
			return UT_ERROR;
		}

		if ((repl = strdup(rep)) == NULL) {
			return UT_ERROR;
		}

		/* Replace '$x' in replacement string with substring.
		 */
		c = repl;
		replaced = false;
		while (*c != '\0') {
			if (*c == '$') {
				if ((*(c+1) >= '0') && (*(c+1) <= '9')) {
					i = *(c+1) - 48;
					if (pmatch[i].rm_so != -1) {
						len = pmatch[i].rm_eo - pmatch[i].rm_so;
						if ((sub = strdup(url + pmatch[i].rm_so)) == NULL) {
							free(repl);
							return UT_ERROR;
						}
						sub[len] = '\0';
					} else {
						sub = strdup("");
					}
					n = c - repl;

					if (replace(repl, n, 2, sub, &tmp) == -1) {
						free(repl);
						return UT_ERROR;
					}

					if (replaced) {
						free(repl);
					}
					repl = tmp;
					c = repl + n + strlen(sub) - 1;
					replaced = true;
					free(sub);
				}
			}
			c++;
		}

		/* Replace pattern with replacement string.
		 */
		len = pmatch[0].rm_eo - ofs;
		len_rep = strlen(repl);
		if (replace(url, ofs, len, repl, new_url) == -1) {
			free(repl);
			return UT_ERROR;
		}
		url = *new_url;

		free(repl);
	}

	return 0;
}

int use_toolkit(char *url, char *toolkit_id, t_url_toolkit *url_toolkit, char **new_url, char *website_root, char **fastcgi_server) {
	t_url_toolkit *toolkit;
	t_toolkit_rule *rule;
	bool replaced = false;
	int loop = 0, result, skip = 0;
	t_fsbool is_dir;
	char *file, *qmark;

	*new_url = NULL;
	*fastcgi_server = NULL;

	if ((toolkit = find_toolkit(toolkit_id, url_toolkit)) == NULL) {
		return UT_ERROR;
	}

	rule = toolkit->toolkit_rule;
	while (rule != NULL) {
		if (skip > 0) {
			skip--;
		} else switch (rule->action) {
			case ta_match_rewrite:
				if (do_rewrite(url, &(rule->pattern), rule->parameter, new_url, rule->match_loop) == UT_ERROR) {
					if (*new_url != NULL) {
						free(*new_url);
						*new_url = NULL;
					}
					return UT_ERROR;
				}
				if (*new_url != NULL) {
					if (replaced) {
						free(url);
					}
					url = *new_url;
					replaced = true;

					if (rule->done == td_return) {
						return UT_RETURN;
					} else if (rule->done == td_exit) {
						return UT_EXIT;
					}
				} else if (replaced) {
					*new_url = url;
				}
				break;
			case ta_match_redirect:
				if (do_rewrite(url, &(rule->pattern), rule->parameter, new_url, rule->match_loop) == UT_ERROR) {
					if (*new_url != NULL) {
						free(*new_url);
						*new_url = NULL;
					}
					return UT_ERROR;
				}
				if (*new_url != NULL) {
					if (replaced) {
						free(url);
					}
					return UT_REDIRECT;
				} else if (replaced) {
					*new_url = url;
				}
				break;
			case ta_call:
			case ta_match_call:
			case ta_match_goto:
				if (rule->action != ta_call) {
					if (regexec(&(rule->pattern), url, 0, NULL, 0) == REG_NOMATCH) {
						break;
					}
				}
				if (++loop == MAX_RULE_LOOP) {
					return UT_ERROR;
				}

				if ((result = use_toolkit(url, rule->parameter, url_toolkit, new_url, website_root, fastcgi_server)) == UT_ERROR) {
					if (*new_url != NULL) {
						free(*new_url);
						*new_url = NULL;
					}
					return UT_ERROR;
				}

				if (*new_url != NULL) {
					if (replaced) {
						free(url);
					}
					url = *new_url;
					replaced = true;
				}

				if (result == UT_EXIT) {
					return UT_EXIT;
				} else if (rule->action == ta_match_goto) {
					return UT_RETURN;
				}
				break;
			case ta_match_deny_access:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_DENY_ACCESS;
				}
				break;
			case ta_match_exit:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_EXIT;
				}
				break;
			case ta_match_fastcgi:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					*fastcgi_server = rule->parameter;
					return UT_EXIT;
				}
				break;
			case ta_match_return:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_RETURN;
				}
				break;
			case ta_match_skip:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					skip = rule->value;
				}
				break;
			case ta_skip:
				skip = rule->value;
				break;
			case ta_requesturi:
				if (valid_uri(url) == false) {
					break;
				}
				if ((file = make_path(website_root, url)) == NULL) {
					return UT_ERROR;
				}

				if ((qmark = strchr(file, '?')) != NULL) {
					*qmark = '\0';
				}
				url_decode(file);
				is_dir = is_directory(file);
				free(file);

				switch (rule->value) {
					case IU_EXISTS:
						if ((is_dir == yes) || (is_dir == no)) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
					case IU_ISFILE:
						if (is_dir == no) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
					case IU_ISDIR:
						if (is_dir == yes) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
				}
				break;
		}
		rule = rule->next;
	}

	return UT_RETURN;
}

#endif
