/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License. For a copy,
 * see http://www.gnu.org/licenses/gpl-2.0.html.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include "config.h"

#ifdef HAVE_TOOLKIT

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "toolkit.h"
#include "libstr.h"
#include "libfs.h"
#include "alternative.h"

#define MAX_RULE_LOOP   20
#define MAX_MATCH_LOOP  20

t_url_toolkit *find_toolkit(char *toolkit_id, t_url_toolkit *url_toolkit) {
	if (toolkit_id == NULL) {
		return NULL;
	}

	while (url_toolkit != NULL) {
		if (strcmp(url_toolkit->toolkit_id, toolkit_id) == 0) {
			return url_toolkit;
		}
		url_toolkit = url_toolkit->next;
	}

	return NULL;
}

static int replace(char *src, int ofs, int len, char *rep, char **dst) {
	int len_rep;

	if ((src == NULL) || (rep == NULL) || (dst == NULL)) {
		return -1;
	}

	len_rep = strlen(rep);
	if ((*dst = (char*)malloc(strlen(src) - len + len_rep + 1)) == NULL) {
		return -1;
	}

	memcpy(*dst, src, ofs);
	memcpy(*dst + ofs, rep, len_rep);
	strcpy(*dst + ofs + len_rep, src + ofs + len);

	return 0;
}

bool toolkit_setting(char *key, char *value, t_url_toolkit *toolkit) {
	t_toolkit_rule *new_rule, *rule;
	char *rest;
	int loop, time;

	if ((key == NULL) || (value == NULL) || (toolkit == NULL)) {
		return false;
	}

	if (strcmp(key, "toolkitid") == 0) {
		if ((toolkit->toolkit_id = strdup(value)) != NULL) {
			return true;
		}
	} else {
		if ((new_rule = (t_toolkit_rule*)malloc(sizeof(t_toolkit_rule))) == NULL) {
			return false;
		} else if (toolkit->toolkit_rule == NULL) {
			toolkit->toolkit_rule = new_rule;
		} else {
			rule = toolkit->toolkit_rule;
			while (rule->next != NULL) {
				rule = rule->next;
			}
			rule->next = new_rule;
		}
		new_rule->parameter = NULL;
		new_rule->done = td_continue;
		new_rule->match_loop = 1;
		new_rule->next = NULL;

		if (strcmp(key, "match") == 0) {
			/* Match
			 */
			if (split_string(value, &value, &rest, ' ') == -1) {
				return false;
			} else if (regcomp(&(new_rule->pattern), value, REG_EXTENDED) != 0) {
				return false;
			}
			split_string(rest, &value, &rest, ' ');
			
			if (strcasecmp(value, "ban") == 0) {
				/* Match UseFastCGI
				 */
				new_rule->action = ta_match_ban;
				if ((new_rule->value = str2int(rest)) == false) {
					return false;
				}
			} else if (strcasecmp(value, "call") == 0) {
				/* Match Call
				 */
				new_rule->action = ta_match_call;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "denyaccess") == 0) {
				/* Match DenyAccess
				 */
				new_rule->action = ta_match_deny_access;
			} else if (strcasecmp(value, "exit") == 0) {
				/* Match Exit
				 */
				new_rule->action = ta_match_exit;
			} else if (strcasecmp(value, "expire") == 0) {
				/* Match Expire
				 */
				new_rule->action = ta_match_expire;
				if (split_string(rest, &value, &rest, ' ') == -1) {
					return false;
				}
				if ((new_rule->value = str2int(value)) == -1) {
					return false;
				}

				time = new_rule->value;

				split_string(rest, &value, &rest, ' ');
				if (strcasecmp(value, "minutes") == 0) {
					new_rule->value *= MINUTE;
				} else if (strcasecmp(value, "hours") == 0) {
					new_rule->value *= HOUR;
				} else if (strcasecmp(value, "days") == 0) {
					new_rule->value *= DAY;
				} else if (strcasecmp(value, "weeks") == 0) {
					new_rule->value *= 7 * DAY;
				} else if (strcasecmp(value, "months") == 0) {
					new_rule->value *= 30.5 * DAY;
				} else if (strcasecmp(value, "seconds") != 0) {
					return false;
				}

				if (new_rule->value < time) {
					return false;
				}

				if (rest == NULL) {
					new_rule->done = td_continue;
				} else if (strcasecmp(rest, "exit") == 0) {
					new_rule->done = td_exit;
				} else if (strcasecmp(rest, "return") == 0) {
					new_rule->done = td_return;
				} else {
					return false;
				}
			} else if (strcasecmp(value, "goto") == 0) {
				/* Match Goto
				 */
				new_rule->action = ta_match_goto;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "redirect") == 0) {
				/* Match Redirect
				 */
				new_rule->action = ta_match_redirect;
				if (strlen(rest) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(rest)) == NULL) {
					return false;
				}
			} else if (strcasecmp(value, "return") == 0) {
				/* Match Return
				 */
				new_rule->action = ta_match_return;
			} else if (strcasecmp(value, "rewrite") == 0) {
				/* Match Rewrite
				 */
				new_rule->action = ta_match_rewrite;
				new_rule->done = td_exit;

				split_string(rest, &value, &rest, ' ');
				if (strlen(value) == 0) {
					return false;
				} else if ((new_rule->parameter = strdup(value)) == NULL) {
					return false;
				}

				if (rest != NULL) {
					split_string(rest, &value, &rest, ' ');
					if ((loop = str2int(value)) > 0) {
						if (loop > MAX_MATCH_LOOP) {
							return false;
						} else if ((value = rest) == NULL) {
							return false;
						}
						new_rule->match_loop = loop;
					} else if (rest != NULL) {
						return false;
					}

					if (strcasecmp(value, "continue") == 0) {
						new_rule->done = td_continue;
					} else if (strcasecmp(value, "return") == 0) {
						new_rule->done = td_return;
					} else {
						return false;
					}
				}
			} else if (strcasecmp(value, "skip") == 0) {
				/* Match Skip
				 */
				new_rule->action = ta_match_skip;
				if ((new_rule->value = str2int(rest)) < 1) {
					return false;
				}
			} else if (strcasecmp(value, "usefastcgi") == 0) {
				/* Match UseFastCGI
				 */
				new_rule->action = ta_match_fastcgi;
				new_rule->parameter = strdup(rest);
			} else {
				return false;
			}
		} else if (strcmp(key, "call") == 0) {
			/* Call
			 */
			new_rule->action = ta_call;
			if (strlen(value) == 0) {
				return false;
			} else if ((new_rule->parameter = strdup(value)) == NULL) {
				return false;
			}
		} else if (strcmp(key, "skip") == 0) {
			/* Skip
			 */
			new_rule->action = ta_skip;
			if ((new_rule->value = str2int(value)) < 1) {
				return false;
			}
		} else if (strcmp(key, "requesturi") == 0) {
			/* RequestURI
			 */
			new_rule->action = ta_requesturi;
			if (split_string(value, &value, &rest, ' ') == -1) {
				return false;
			}
			if (strcasecmp(value, "exists") == 0) {
				new_rule->value = IU_EXISTS;
			} else if (strcasecmp(value, "isfile") == 0) {
				new_rule->value = IU_ISFILE;
			} else if (strcasecmp(value, "isdir") == 0) {
				new_rule->value = IU_ISDIR;
			} else {
				return false;
			}
			if (strcasecmp(rest, "return") == 0) {
				new_rule->done = td_return;
			} else if (strcasecmp(rest, "exit") == 0) {
				new_rule->done = td_exit;
			} else {
				return false;
			}
		} else if (strcmp(key, "usessl") == 0) {
			/* UseSSL
			 */
		new_rule->action = ta_usessl;

		} else {
			return false;
		}

		return true;
	}

	return false;
}

bool toolkit_rules_oke(t_url_toolkit *url_toolkit) {
	t_url_toolkit *toolkit;
	t_toolkit_rule *rule;

	toolkit = url_toolkit;
	while (toolkit != NULL) {
		if (toolkit->toolkit_id == NULL) {
			fprintf(stderr, "A ToolkitID is missing in an UrlToolkit section.\n");
			return false;
		}

		rule = toolkit->toolkit_rule;
		while (rule != NULL) {
			if ((rule->action == ta_match_goto) || (rule->action == ta_match_call) || (rule->action == ta_call)) {
				if (rule->parameter == NULL) {
					fprintf(stderr, "Missing parameter in toolkit rule '%s'.\n", toolkit->toolkit_id);
					return false;
				} else if (find_toolkit(rule->parameter, url_toolkit) == NULL) {
					fprintf(stderr, "Unknown ToolkitID in Goto/Call in toolkit rule '%s'.\n", toolkit->toolkit_id);
					return false;
				}
			}
			rule = rule->next;
		}
		toolkit = toolkit->next;
	}

	return true;
}

static int do_rewrite(char *url, regex_t *regexp, char *rep, char **new_url, int loop) {
	int ofs, len, len_rep, i, n;
	regmatch_t pmatch[10];
	char *repl, *c, *sub, *tmp;
	bool replaced;

	if ((url == NULL) || (regexp == NULL) || (rep == NULL) || (new_url == NULL)) {
		return -1;
	}

	*new_url = NULL;
	while (loop-- > 0) {
		if (regexec(regexp, url, 10, pmatch, 0) == REG_NOMATCH) {
			break;
		}
		if ((ofs = pmatch[0].rm_so) == -1) {
			return -1;
		}

		if ((repl = strdup(rep)) == NULL) {
			return -1;
		}

		/* Replace '$x' in replacement string with substring.
		 */
		c = repl;
		replaced = false;
		while (*c != '\0') {
			if (*c == '$') {
				if ((*(c+1) >= '0') && (*(c+1) <= '9')) {
					i = *(c+1) - 48;
					if (pmatch[i].rm_so != -1) {
						len = pmatch[i].rm_eo - pmatch[i].rm_so;
						if ((sub = strdup(url + pmatch[i].rm_so)) == NULL) {
							free(repl);
							return -1;
						}
						sub[len] = '\0';
					} else {
						sub = strdup("");
					}
					n = c - repl;

					if (replace(repl, n, 2, sub, &tmp) == -1) {
						free(repl);
						return -1;
					}

					if (replaced) {
						free(repl);
					}
					repl = tmp;
					c = repl + n + strlen(sub) - 1;
					replaced = true;
					free(sub);
				}
			}
			c++;
		}

		/* Replace pattern with replacement string.
		 */
		len = pmatch[0].rm_eo - ofs;
		len_rep = strlen(repl);
		if (replace(url, ofs, len, repl, new_url) == -1) {
			free(repl);
			return -1;
		}
		url = *new_url;

		free(repl);
	}

	return 0;
}

void init_toolkit_options(t_toolkit_options *options, char *website_root, t_url_toolkit *toolkit, bool use_ssl) {
	options->new_url = NULL;
	options->website_root = website_root;
	options->fastcgi_server = NULL;
	options->ban = 0;
	options->expire = -1;
	options->url_toolkit = toolkit;
	options->use_ssl = use_ssl;
}

int use_toolkit(char *url, char *toolkit_id, t_toolkit_options *options) {
	t_url_toolkit *toolkit;
	t_toolkit_rule *rule;
	bool replaced = false;
	int loop = 0, result, skip = 0;
	t_fsbool is_dir;
	char *file, *qmark;

	if (options == NULL) {
		return UT_ERROR;
	}

	options->new_url = NULL;

	if ((toolkit = find_toolkit(toolkit_id, options->url_toolkit)) == NULL) {
		return UT_ERROR;
	}

	rule = toolkit->toolkit_rule;
	while (rule != NULL) {
		if (skip > 0) {
			skip--;
		} else switch (rule->action) {
			case ta_match_rewrite:
				if (do_rewrite(url, &(rule->pattern), rule->parameter, &(options->new_url), rule->match_loop) == -1) {
					if (options->new_url != NULL) {
						free(options->new_url);
						options->new_url = NULL;
					}
					return UT_ERROR;
				}
				if (options->new_url != NULL) {
					if (replaced) {
						free(url);
					}
					url = options->new_url;
					replaced = true;

					if (rule->done == td_return) {
						return UT_RETURN;
					} else if (rule->done == td_exit) {
						return UT_EXIT;
					}
				} else if (replaced) {
					options->new_url = url;
				}
				break;
			case ta_match_redirect:
				if (do_rewrite(url, &(rule->pattern), rule->parameter, &(options->new_url), rule->match_loop) == -1) {
					if (options->new_url != NULL) {
						free(options->new_url);
						options->new_url = NULL;
					}
					return UT_ERROR;
				}
				if (options->new_url != NULL) {
					if (replaced) {
						free(url);
					}
					return UT_REDIRECT;
				} else if (replaced) {
					options->new_url = url;
				}
				break;
			case ta_call:
			case ta_match_call:
			case ta_match_goto:
				if (rule->action != ta_call) {
					if (regexec(&(rule->pattern), url, 0, NULL, 0) == REG_NOMATCH) {
						break;
					}
				}
				if (++loop == MAX_RULE_LOOP) {
					return UT_ERROR;
				}

				if ((result = use_toolkit(url, rule->parameter, options)) == UT_ERROR) {
					if (options->new_url != NULL) {
						free(options->new_url);
						options->new_url = NULL;
					}
					return UT_ERROR;
				}

				if (options->new_url != NULL) {
					if (replaced) {
						free(url);
					}
					url = options->new_url;
					replaced = true;
				} else if (replaced) {
					options->new_url = url;
				}

				if ((result == UT_DENY_ACCESS) || (result == UT_EXIT) || (result == UT_REDIRECT)) {
					return result;
				} else if (rule->action == ta_match_goto) {
					return UT_RETURN;
				}
				break;
			case ta_match_ban:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					options->ban = rule->value;
					return UT_EXIT;
				}
				break;
			case ta_match_deny_access:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_DENY_ACCESS;
				}
				break;
			case ta_match_exit:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_EXIT;
				}
				break;
			case ta_match_expire:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					options->expire = rule->value;

					if (rule->done == td_return) {
						return UT_RETURN;
					} else if (rule->done == td_exit) {
						return UT_EXIT;
					}
				}
				break;
			case ta_match_fastcgi:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					options->fastcgi_server = rule->parameter;
					return UT_EXIT;
				}
				break;
			case ta_match_return:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					return UT_RETURN;
				}
				break;
			case ta_match_skip:
				if (regexec(&(rule->pattern), url, 0, NULL, 0) == 0) {
					skip = rule->value;
				}
				break;
			case ta_skip:
				skip = rule->value;
				break;
			case ta_requesturi:
				if (valid_uri(url) == false) {
					break;
				}
				if ((file = make_path(options->website_root, url)) == NULL) {
					return UT_ERROR;
				}

				if ((qmark = strchr(file, '?')) != NULL) {
					*qmark = '\0';
				}
				url_decode(file);
				is_dir = is_directory(file);
				free(file);

				switch (rule->value) {
					case IU_EXISTS:
						if ((is_dir == yes) || (is_dir == no)) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
					case IU_ISFILE:
						if (is_dir == no) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
					case IU_ISDIR:
						if (is_dir == yes) {
							return (rule->done == td_return ? UT_RETURN : UT_EXIT);
						}
						break;
				}
				break;
			case ta_usessl:
				if (options->use_ssl == false) {
					break;
				}
				switch (rule->value)
				break;
		}
		rule = rule->next;
	}

	return UT_RETURN;
}

#endif
