<?php
	class ACMEv1 {
		private $server = null;
		private $hostname = null;
		private $account_key = null;
		private $nonce = null;
		private $ca_terms = null;
		private $first_request = true;

		/* Constructor
		 */
		public function __construct($hostname, $account_key) {
			$this->server = new HTTPS($hostname);

			$this->hostname = $hostname;
			$this->account_key = $account_key;

			if (($result = $this->GET("/directory")) != false) {
				$this->nonce = $result["headers"]["replay-nonce"];
				if (($props = json_decode($result["body"], true)) !== null) {
					$this->ca_terms = $props["meta"]["terms-of-service"];
				}
			}
		}

		/* Base64 URL safe encoding
		 */
		private function b64u_encode($string) {
			return str_replace("=", "", strtr(base64_encode($string), "+/", "-_"));
		}

		/* Base64 URL safe decoding
		 */
		private function b64u_decode($string) {
			$padding = strlen($input) % 4;
			if ($padding > 0) {
				$padding = 4 - $padding;
				$input .= str_repeat("=", $padding);
			}

			return base64_decode(strtr($string, "-_", "+/"));
		}

		/* Get path part from URI
		 */
		private function get_path($uri) {
			list(,, $hostname, $path) = explode("/", $uri, 4);
			if ($hostname != $this->hostname) {
				return false;
			}
			list($path) = explode(">", $path, 2);

			return "/".$path;
		}

		/* Send API GET request
		 */
		public function GET($uri, $expected_status = 200) {
			if (($result = $this->server->GET($uri)) === false) {
				return false;
			}

			if ($result["status"] != $expected_status) {
				return false;
			}

			return $result;
		}

		/* Send API POST request
		 */
		public function POST($uri, $payload, $expected_status = 200) {
			$protected = $this->b64u_encode(json_encode(array(
				"alg"   => "RS256",
				"jwk"   => array(
					"kty" => "RSA",
					"e"   => $this->b64u_encode($this->account_key->e),
					"n"   => $this->b64u_encode($this->account_key->n)),
				"nonce" => $this->nonce)));
			$payload = $this->b64u_encode(str_replace('\\/', '/', json_encode($payload)));

			openssl_sign($protected.".".$payload, $signature, $this->account_key->private_key, "SHA256");
			$signature = $this->b64u_encode($signature);

			$data = json_encode(array(
				"protected" => $protected,
				"payload"   => $payload,
				"signature" => $signature));

			$this->server->add_header("Accept", "application/json");
			$this->server->add_header("Content-Type", "application/json");

			if (($result = $this->server->POST($uri, $data)) === false) {
				printf(" - HTTP error for %s.\n", $uri);
				return false;
			} else if ($result["status"] != $expected_status) {
				if (($body = json_decode($result["body"], true)) !== null) {
					printf(" - %s\n", $body["detail"]);
				}
				return false;
			}

			$this->nonce = $result["headers"]["replay-nonce"];

			/* Follow Link header
			 */
			if (($link = $result["headers"]["link"]) != null) {
				if (($path = $this->get_path($link)) != false) {
					$this->GET($path);
				}
			}

			return $result;
		}

		/* Register account
		 */
		public function register_account($email_address) {
			$payload = array(
				"resource"  => "new-reg",
				"contact"   => array("mailto:".$email_address),
				"agreement" => $this->ca_terms);

			if (($result = $this->POST("/acme/new-reg", $payload, 201)) == false) {
				return false;
			}

			return true;
		}

		/* Get information for HTTP challenge
		 */
		private function get_http_authentication_challenge($website_hostname) {
			$payload = array(
				"resource"   => "new-authz",
				"identifier" => array(
					"type"  => "dns",
					"value" => $website_hostname));

			if (($result = $this->POST("/acme/new-authz", $payload, 201)) == false) {
				return false;
			}

			$body = json_decode($result["body"], true);
			foreach ($body["challenges"] as $challenge) {
				if ($challenge["type"] == "http-01") {
					return $challenge;
				}
			}

			printf(" - No HTTP authentication challenge received.\n");

			return false;
		}

		/* Get authorization key
		 */
		public function get_authorization_key($website_root, $challenge) {
			/* Create verification file
			 */
			$data = array(
				"e"   => $this->b64u_encode($this->account_key->e),
				"kty" => "RSA",
				"n"   => $this->b64u_encode($this->account_key->n));

			$content = $challenge["token"].".".$this->b64u_encode(hash("sha256", json_encode($data), true));

			$dir = $website_root."/.well-known/acme-challenge";
			if (file_exists($dir) == false) {
				if (mkdir($dir, 0755, true) == false) {
					printf(" - Can't create directory %s.\n", $dir);
					return false;
				}
			}
			if (file_put_contents($dir."/".$challenge["token"], $content) === false) {
				printf(" - Can't create token %s/%s.\n", $dir, $challenge["token"]);
				return false;
			}

			$payload = array(
				"resource"         => "challenge",
				"type"             => "http-01",
				"keyAuthorization" => $content,
				"token"            => $challenge["token"]);

			/* Send response
			 */
			if (($path = $this->get_path($challenge["uri"])) == false) {
				printf(" - No path detected in challenge URI.\n");
				return false;
			}

			if (($result = $this->POST($path, $payload, 202)) == false) {
				return false;
			}
			$body = json_decode($result["body"], true);

			/* Wait for acceptance
			 */
			if (($path = $this->get_path($body["uri"])) == false) {
				printf(" - No path found in acceptance URL.\n");
				return false;
			}
			do {
				if (($result = $this->server->GET($path, 202)) === false) {
					printf(" - HTTP error while waiting for acceptance.\n");
					return false;
				} else if ($result["status"] != 202) {
					$body = json_decode($result["body"], true);
					printf(" - %s.\n", $body["detail"]);
					return false;
				}
				$body = json_decode($result["body"], true);
				sleep(1);
			} while ($body["status"] == "pending");

			if ($body["status"] != "valid") {
				printf(" - %s.\n", $body["error"]["detail"]);
				printf(" - No valid authorization key received.\n");
				return false;
			}

			return $body;
		}

		public function authorize_hostname($website_hostname, $website_root) {
			printf("Authorizing %s.\n", $website_hostname);

			/* Get HTTP authentication challenge
			*/
			printf(" - Retrieving HTTP authentication challenge.\n");
			$challenge = $this->get_http_authentication_challenge($website_hostname);
			if ($challenge === false) {
				printf(" - Authentication token for HTTP challenge not found.\n");
				return false;
			}

			/* Get authorization key
			*/
			printf(" - Retrieving authorization key.\n");
			$authorization = $this->get_authorization_key($website_root, $challenge);

			/* Clean up
			 */
			$subdir = "/.well-known/acme-challenge";
			if (($dp = opendir($website_root.$subdir)) !== false) {
				while (($file = readdir($dp)) !== false) {
					if (substr($file, 0, 1) != ".") {
						unlink($website_root.$subdir."/".$file);
					}
				}
				closedir($dp);
			}
			rmdir($website_root.$subdir);
			rmdir($website_root."/.well-known");

			if ($authorization == false) {
				printf(" - Error while retrieving authorization key.\n");
				return false;
			}

			return true;
		}

		/* Get certificate
		 */
		public function get_certificate($csr, $authorization) {
			$payload = array(
				"resource" => "new-cert",
				"csr"      => $this->b64u_encode($csr));
			if (($result = $this->POST("/acme/new-cert", $payload, 201)) == false) {
				return false;
			}

			return $result["body"];
		}

		/* Revoke certificate
		 */
		public function revoke_certificate($cert) {
			$payload = array(
				"resource"    => "revoke-cert",
				"certificate" => $this->b64u_encode($cert));
			if (($result = $this->POST("/acme/revoke-cert", $payload)) == false) {
				return false;
			}

			return true;
		}
	}
?>
