<?php
	/* libraries/general.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	/* Flatten array to new array with depth 1
	 *
	 * INPUT:  array
	 * OUTPUT: array
	 * ERROR:  -
	 */
	function array_flatten($data) {
		$result = array();
		foreach ($data as $item) {
			if (is_array($item)) {
				$result = array_merge($result, array_flatten($item));
			} else {
				array_push($result, $item);
			}
		}

		return $result;
	}

	/* Match if IP matches IP subnet
	 *
	 * INPUT:  string ip, string net
	 * OUTPUT: bool
	 * ERROR:  -
	 */
	function ip_match($ip, $net) {
		list($net, $mask) = explode("/", $net, 2);
		if ($mask == "") {
			$mask = 0;
		} else {
			$mask = 32 - (int)$mask;
		}

		$ip = explode(".", $ip);
		$ip = ((int)$ip[0] << 24) + ((int)$ip[1] << 16) + ((int)$ip[2] << 8) + (int)$ip[3];
		$ip = $ip >> $mask;

		$net = explode(".", $net);
		$net = ((int)$net[0] << 24) + ((int)$net[1] << 16) + ((int)$net[2] << 8) + (int)$net[3];
		$net = $net >> $mask;

		return $ip == $net;
	}

	/* Prepare string for unescaped output
	 *
	 * INPUT:  string data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	function unescaped_output($str) {
		$str = htmlentities($str);
		$str = str_replace("\r", "", $str);
		$str = str_replace("\n", "<br />", $str);

		return $str;
	}

	/* Prepare string for iCalendar output
	 *
	 * INPUT:  string data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	function ics_output($str) {
		$str = str_replace("\\", "\\\\", $str);
		$str = str_replace("\"", "\\\"", $str);
		$str = str_replace("\r", "", $str);
		$str = str_replace("\n", "\\n", $str);

		return $str;
	}

	/* Generate link from text
	 *
	 * INPUT:  string text
	 * OUTPUT: string link
	 * ERROR:  -
	 */
	function make_link($text) {
		$result = "";
		$len = strlen($text);
		for ($i = 0; $i < $len; $i++) {
			if (($text[$i] >= "a") && ($text[$i] <= "z")) {
				$result .= $text[$i];
			} else if (($text[$i] >= "A") && ($text[$i] <= "Z")) {
				$result .= strtolower($text[$i]);
			} else if (($text[$i] >= "0") && ($text[$i] <= "9")) {
				$result .= $text[$i];
			} else if ($text[$i] == " ") {
				$result .= "-";
			}
		}

		return $result;
	}

	/* Truncate text
	 *
	 * INPUT:  string text, int length
	 * OUTPUT: string truncated text
	 * ERROR:  -
	 */
	function truncate_text($text, $length) {
		if (strlen($text) <= $length) {
			return $text;
		}

		$is_space = ($text[$length] === " ");
		$text = substr($text, 0, $length);
		if ($is_space == false) {
			if (($pos = strrpos($text, " ")) !== false) {
				$text = substr($text, 0, $pos);
			}
		}

		return $text."...";
	}

	/* Truncate HTML
	 *
	 * INPUT:  string HTML, int length
	 * OUTPUT: string truncated HTML
	 * ERROR:  -
	 */
	function truncate_html($html, $length) {
		$open_tags = array();
		$html_len = strlen($html);

		for ($i = 0; $i < $html_len; $i++) {
			if ($html[$i] == "<") {
				$name_begin = $i + 1;
				if (($tag_end = strpos($html, ">", $name_begin)) === false) {
					continue;
				}
				$i = $tag_end;

				if ($html[$tag_end - 1] == "/") {
					continue;
				}

				if ($open_tag = ($html[$name_begin] == "/")) {
					array_pop($open_tags);
				} else {
					if (($name_end = strpos($html, " ", $name_begin)) === false) {
						$name_end = $tag_end;
					} else if ($name_end > $tag_end) {
						$name_end = $tag_end;
					}

					$tag = substr($html, $name_begin, $name_end - $name_begin);
					array_push($open_tags, $tag);
				}
			} else if (--$length == 0) {
				break;
			}
		}

		$html = substr($html, 0, $i + 1);
		if ($length == 0) {
			$html .= "...";
		}

		$open_tags = array_reverse($open_tags);
		foreach ($open_tags as $tag) {
			$html .= "</".$tag.">";
		}

		return $html;
	}

	/* Decode a GZip encoded string
	 *
	 * INPUT:  string
	 * OUTPUT: string
	 * ERROR:  -
	 */
	if (version_compare(PHP_VERSION, "6.0.0", "<")) {
		function gzdecode($data) {
			$file = tempnam("/tmp", "gzip");

			@file_put_contents($file, $data);
			ob_start();
			readgzfile($file);
			$data = ob_get_clean();
			unlink($file);

			return $data;
		}
	}
?>
