<?php
	/* libraries/http.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	class HTTP {
		private $cookies = array();
		protected $default_port = 80;
		protected $protocol = "";

		/* Connect to webserver
		 *
		 * INPUT:  string hostname, int port number
		 * OUTPUT: resource socket
		 * ERROR:  false
		 */
		protected function connect($host, $port) {
			return fsockopen($this->protocol.$host, $port);
		}

		protected function rename_host($host) {
			return $host;
		}

		/* Perform HTTP request
		 *
		 * INPUT:  string host, array HTTP header[, string request body]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		protected function request($host, $header, $body = "", $redirect = 10) {
			list($host, $port) = explode(":", $host);
			if ($port === null) {
				$port = $this->default_port;
			}

			if (($sock = $this->connect($host, $port)) == false) {
				return false;
			}
			array_push($header, "Host: ".$this->rename_host($host));
			array_push($header, "Accept-Encoding: gzip");

			$cookies = array();
			if (is_array($this->cookies[$host])) {
				foreach ($this->cookies[$host] as $key => $value) {
					array_push($cookies, $key."=".$value);
				}
				if (count($cookies) > 0) {
					array_push($header, "Cookie: ".implode("; ", $cookies));
				}
			} else {
				$this->cookies[$host] = array();
			}

			array_push($header, "User-Agent: Mozilla/5.0 (compatible; Banshee HTTP class)");

			$request = implode("\r\n", $header)."\r\n\r\n".$body;
			fputs($sock, $request);

			/* Read response
			 */
			$result = "";
			while (($line = fgets($sock)) !== false) {
				$result .= $line;
			}
			fclose($sock);

			/* Build response array
			 */
			list($header, $body) = explode("\r\n\r\n", $result, 2);
			$header = explode("\r\n", $header);
			list(, $status) = explode(" ", $header[0]);

			$response["status"] = (int)$status;
			$response["header"] = array();
			$response["body"] = $body;

			for ($i = 1; $i < count($header); $i++) {
				list($key, $value) = explode(":", $header[$i], 2);
				$key = trim($key);
				$value = trim($value);
				if ($key == "Set-Cookie") {
					/* Cookie
					 */
					list($value) = explode(";", $value);
					list($cookie_key, $cookie_value) = explode("=", $value);
					$this->cookies[$host][$cookie_key] = $cookie_value;
				} else if ($key == "Content-Encoding") {
					/* GZip content encoding
					 */
					if (strpos($value, "gzip") !== false) {
						$response["body"] = gzdecode($response["body"]);
					}
				} else {
					/* Other HTTP header
					 */
					$response["header"][$key] = $value;
				}
			}

			if (($redirect > 0) && (($status == "301") || ($status == "302"))) {
				if (isset($response["header"]["Location"])) {
					$location = $response["header"]["Location"];

					return $this->request($host, $header, $body, $redirect - 1);
				}
			}

			return $response;
		}

		/* Perform HTTP GET request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function GET($host, $uri) {
			$header = array("GET ".$uri." HTTP/1.0");

			return $this->request($host, $header);
		}

		/* Perform HTTP HEAD request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function HEAD($host, $uri) {
			$header = array("HEAD ".$uri." HTTP/1.0");

			return $this->request($host, $header);
		}

		/* Perform HTTP POST request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function POST($host, $uri, $data) {
			if (is_array($data) == false) {
				$body = $data;
			} else {
				$body = array();
				foreach ($data as $key => $value) {
					array_push($body, urlencode($key)."=".urlencode($value));
				}
				$body = implode("&", $body);
			}

			$header = array();
			array_push($header, "POST ".$uri." HTTP/1.0");
			array_push($header, "Content-Length: ".strlen($body));
			array_push($header, "Content-Type: application/x-www-form-urlencoded");

			return $this->request($host, $header, $body);
		}

		/* Perform HTTP PUT request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function PUT($host, $uri, $data) {
			$header = array();
			array_push($header, "PUT ".$uri." HTTP/1.0");
			array_push($header, "Content-Length: ".strlen($data));

			return $this->request($host, $header, $data);
		}

		/* Perform HTTP DELETE request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function DELETE($host, $uri) {
			$header = array("DELETE ".$uri." HTTP/1.0");

			return $this->request($host, $header);
		}

		/* Perform HTTP OPTIONS request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function OPTIONS($host, $uri) {
			$header = array("OPTIONS ".$uri." HTTP/1.0");

			return $this->request($host, $header);
		}

		/* Perform HTTP TRACE request
		 *
		 * INPUT:  string hostname, string URI
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function TRACE($host, $uri) {
			$header = array("TRACE ".$uri." HTTP/1.0");

			return $this->request($host, $header);
		}
	}

	/* Encrypted HTTP
	 */
	class HTTPS extends HTTP {
		protected $default_port = 443;
		protected $protocol = "tls://";
	}

	/* HTTP proxy
	 */
	class HTTP_proxy extends HTTP {
		protected $host = null;
		protected $port = null;

		/* Constructor
		 *
		 * INPUT:  string hostname, int port number
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($host, $port) {
			$this->host = $host;
			$this->port = $port;
		}

		/* Connect to proxy
		 *
		 * INPUT:  string hostname, int port number
		 * OUTPUT: resource socket
		 * ERROR:  false
		 */
		protected function connect($host, $port) {
			return fsockopen($this->protocol.$this->host, $this->port);
		}

		/* Generate request URI
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: string url
		 * ERROR:  -
		 */
		private function uri($host, $uri, $ssl) {
			return "http".($ssl ? "s" : "")."://".$host.$uri;
		}

		/* Perform HTTP GET request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function GET($host, $uri, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::GET($host, $uri);
		}

		/* Perform HTTP POST request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function POST($host, $uri, $data, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::POST($host, $uri, $data);
		}

		/* Perform HTTP PUT request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function PUT($host, $uri, $data, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::PUT($host, $uri, $data);
		}

		/* Perform HTTP DELETE request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function DELETE($host, $uri, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::DELETE($host, $uri);
		}

		/* Perform HTTP OPTIONS request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function OPTIONS($host, $uri, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::OPTIONS($host, $uri);
		}

		/* Perform HTTP TRACE request
		 *
		 * INPUT:  string hostname, string URI[, boolean use ssl]
		 * OUTPUT: array( "status" => string status, "header" => HTTP header, "body" => request body )
		 * ERROR:  false
		 */
		public function TRACE($host, $uri, $ssl = false) {
			$uri = $this->uri($host, $uri, $ssl);
			return parent::TRACE($host, $uri);
		}
	}

	/* Encrypted HTTP proxy
	 */
	class HTTPS_proxy extends HTTP_proxy {
		protected $protocol = "tls://";
	}
?>
