<?php
	/* libraries/security.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	/* Remove dangerous characters from string
	 *
	 * INPUT:  string text
	 * OUTPUT: string text
	 * ERROR:  -
	 */
	function secure_input($data) {
		if (is_array($data) == false) {
			$data = utf8_decode($data);
			if (get_magic_quotes_gpc() == 1) {
				$data = stripslashes($data);
			}
			$data = str_replace(chr(0), "", $data);
			$special_chars = "/[".chr(1)."-".chr(8)."]|".
			                  "[".chr(11).chr(12)."]|".
			                  "[".chr(14)."-".chr(31)."]/";
			$data = preg_replace($special_chars, "", $data);
		} else foreach ($data as $key => $value) {
			$data[$key] = secure_input($value);
		}

		return $data;
	}

	/* Exit if client's IP has been banned
	 *
	 * INPUT:  array/string banned ips
	 * OUTPUT: -
	 * ERROR:  -
	 */
	function check_ip_bans($banned_ips) {
		$banned_ips = explode(",", $banned_ips);

		foreach ($banned_ips as $banned_ip) {
			if (($banned_ip = trim($banned_ip)) == "") {
				continue;
			}
			if (ip_match($_SERVER["REMOTE_ADDR"], $banned_ip)) {
				header("Status: 403 Forbidden");
				exit;
			}
		}
	}

	/* Validate input
	 *
	 * INPUT:  string input, string valid characters[, int length]
	 * OUTPUT: boolean input oke
	 * ERROR:  -
	 */
	function valid_input($data, $allowed, $length = null) {
		if (is_array($data) == false) {
			$data_len = strlen($data);

			if ($length !== null) {
				if ($length == VALIDATE_NONEMPTY) {
					if ($data_len == 0) {
						return false;
					}
				} else if ($data_len !== $length) {
					return false;
				}
			} else if ($data_len == 0) {
				return true;
			}

			$data = str_split($data);
			$allowed = str_split($allowed);
			$diff = array_diff($data, $allowed);

			return count($diff) == 0;
		} else foreach ($data as $item) {
			if (valid_input($item, $allowed, $length) == false) {
				return false;
			}
		}

		return true;
	}

	/* Validate an e-mail address
	 *
	 * INPUT:  string e-mail address
	 * OUTPUT: boolean e-mail address oke
	 * ERROR:  -
	 */
	function valid_email($email) {
		return eregi("^[0-9a-z]([-_.~]?[0-9a-z])*@[0-9a-z]([-.]?[0-9a-z])*\\.[a-z]{2,4}$", $email) != false;
	}

	/* Validate a date string
	 *
	 * INPUT:  string date
	 * OUTPUT: boolean date oke
	 * ERROR:  -
	 */
	function valid_date($date) {
		return eregi("^[0-9]{4}-[0-9]{2}-[0-9]{2}$", $date) != false;
	}

	/* Validate a time string
	 *
	 * INPUT:  string time
	 * OUTPUT: boolean time oke
	 * ERROR:  -
	 */
	function valid_time($time) {
		return eregi("^(([01]?[0-9])|(2[0-3])):[0-5][0-9](:[0-5][0-9])?$", $time) != false;
	}

	/* Validate a timestamp
	 *
	 * INPUT:  string timestamp
	 * OUTPUT: boolean timestamp oke
	 * ERROR:  -
	 */
	function valid_timestamp($timestamp) {
		list($date, $time) = explode(" ", $timestamp, 2);
		return valid_date($date) && valid_time($time);
	}

	/* Validate a telephone number
	 *
	 * INPUT:  string telephone number
	 * OUTPUT: boolean telephone number oke
	 * ERROR:  -
	 */
	function valid_phonenumber($phonenr) {
		$phonenr = str_replace(" ", "", $phonenr);
		return eregi("^(\+31|0)([0-9]{9}|6-?[0-9]{8}|[0-9]{2}-?[0-9]{7}|[0-9]{3}-?[0-9]{6})$", $phonenr) != false;
	}

	/* Return a per-page overview of the access levels
	 *
	 * INPUT:  object database
	 * OUTPUT: array( string page => int access level[, ....] )
	 * ERROR:  false
	 */
	function page_access_list($db) {
		global $_user;

		$access_rights = array();

		/* Public pages on disk
		 */
		$public = config_file("public_pages");
		foreach ($public as $page) {
			$page = str_replace("*/", "", $page);
			$access_rights[$page] = 1;
		}

		/* Private pages on disk
		 */
		$private = config_file("private_pages");
		foreach ($private as $page) {
			$page = str_replace("*/", "", $page);
			$access_rights[$page] = $_user->is_admin ? YES : NO;
		}
		$access_rights["logout"] = $_user->logged_in ? YES : NO;

		if ($_user->logged_in && ($_user->is_admin == false)) {
			$query = "select * from roles where id in ".
					 "(select role_id from user_role where user_id=%d)";
			if (($roles = $db->execute($query, $_user->id)) === false) {
				return false;
			}
			foreach ($roles as $role) {
				$role = array_slice($role, 2);
				foreach ($role as $page => $level) {
					$level = (int)$level;
					if ($_user->is_admin && ($level == 0)) {
						$level = 1;
					}
					if (isset($access_rights[$page]) == false) {
						$access_rights[$page] = $level;
					} else if ($access_rights[$page] < $level) {
						$access_rights[$page] = $level;
					}
				}
			}
		}

		/* Pages in database
		 */
		if (($pages = $db->execute("select * from pages")) === false) {
			return false;
		}
		foreach ($pages as $page) {
			$access_rights[ltrim($page["url"], "/")] = is_false($page["private"]) || $_user->is_admin ? YES : NO;
		}

		if ($_user->logged_in && ($_user->is_admin == false)) {
			$conditions = $rids = array();
			foreach ($_user->role_ids as $rid) {
				array_push($conditions, "role_id=%d");
				array_push($rids, $rid);
			}

			$query = "select p.url,a.level from pages p, page_access a ".
					 "where p.id=a.page_id and (".implode(" or ", $conditions).")";
			if (($pages = $db->execute($query, $rids)) === false) {
				return false;
			}

			foreach ($pages as $page) {
				$url = ltrim($page["url"], "/");
				if ($access_rights[$url] < $page["level"]) {
					$access_rights[$url] = $page["level"];
				}
			}
		}

		return $access_rights;
	}

	/* Get user's one time key
	 *
	 * INPUT:  object database, int user identifier
	 * OUTPUT: string one time key
	 * ERROR:  false
	 */
	function get_one_time_key($db, $user_id) {
		if (($user = $db->entry("users", $user_id)) == false) {
			return false;
		}

		if ($user["one_time_key"] != null) {
			return $user["one_time_key"];
		}

		$key = md5(microtime().(string)rand());
		if ($db->update("users", $user_id, array("one_time_key" => $key)) == false) {
			return false;
		}

		return $key;
	}
?>
