<?php
	/* libraries/email.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	class email {
		protected $to = array();
		protected $cc = array();
		protected $bcc = array();
		protected $from = null;
		protected $reply_to = null;
		protected $subject = null;
		protected $message = null;
		protected $content_type = "text/plain";
		protected $boundary = null;
		protected $attachments = array();
		protected $sender_address = null;

		/* Constructor
		 *
		 * INPUT:  string subject[, string e-mail][, string name]
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($subject, $from_address = null, $from_name = null) {
			$this->subject = array_shift(explode("\n", $subject));
			if ($this->valid_email($from_address)) {
				$this->from = $this->make_address($from_address, $from_name);
				$this->sender_address = $from_address;
			}
			$this->boundary = substr(md5(time()), 0, 10);
		}

		/* Validate an e-mail address
		*
		* INPUT:  string e-mail address
		* OUTPUT: boolean e-mail address oke
		* ERROR:  -
		*/
		protected function valid_email($email) {
			return eregi("^[0-9a-z]([-_.~]?[0-9a-z])*@[0-9a-z]([-.]?[0-9a-z])*\\.[a-z]{2,4}$", $email) != false;
		}

		/* Combine name and e-mail address
		 *
		 * INPUT:  string e-mail address, string name
		 * OUTPUT: string combined name and address
		 * ERROR:  -
		 */
		protected function make_address($address, $name) {
			$name = array_shift(explode("\n", $name));

			return $name === null ? $address : $name." <".$address.">";
		}

		/* Set reply-to
		 *
		 * INPUT:  string e-mail address[, string name]
		 * OUTPUT: boolean valid e-mail address
		 * ERROR:  -
		 */
		public function reply_to($address, $name = null) {
			if ($this->valid_email($address) == false) {
				return false;
			}

			$this->reply_to = $this->make_address($address, $name);
			$this->sender_address = $address;

			return true;
		}

		/* Add recipient
		 *
		 * INPUT:  string e-mail address[, string name]
		 * OUTPUT: boolean valid e-mail address
		 * ERROR:  -
		 */
		public function to($address, $name = null) {
			if ($this->valid_email($address) == false) {
				return false;
			}

			array_push($this->to, $this->make_address($address, $name));

			return true;
		}

		/* Add recipient from database
		 *
		 * INPUT:  object database, int user id
		 * OUTPUT: boolean valid user id and valid e-mail address
		 * ERROR:  -
		 */
		public function to_user_id($db, $user_id) {
			if (($user = $db->entry("users", $user_id)) == false) {
				return false;
			}

			return $this->to($user["email"], $user["fullname"]);
		}

		/* Add Carbon Copy recipient
		 *
		 * INPUT:  string e-mail address[, string name]
		 * OUTPUT: boolean valid e-mail address
		 * ERROR:  -
		 */
		public function cc($address, $name = null) {
			if ($this->valid_email($address) == false) {
				return false;
			}

			array_push($this->cc, $this->make_address($address, $name));

			return true;
		}

		/* Add Blind Carbon Copy recipient
		 *
		 * INPUT:  string e-mail address[, string name]
		 * OUTPUT: boolean valid e-mail address
		 * ERROR:  -
		 */
		public function bcc($address, $name = null) {
			if ($this->valid_email($address) == false) {
				return false;
			}

			array_push($this->bcc, $this->make_address($address, $name));

			return true;
		}

		/* Set e-mail message
		 *
		 * INPUT:  string message
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function message($message) {	
			/* Determine message mimetype
			 */
			if ((substr($message, 0, 5) == "<body") && (substr(rtrim($message), -7) == "</body>")) {
				$this->content_type = "text/html";
			}

			$this->message = $message;
		}

		/* Add e-mail attachment
		 *
		 * INPUT:  string filename[, string content][, string content type]
		 * OUTPUT: boolean attaching successful
		 * ERROR:  -
		 */
		public function attachment($filename, $content = null, $content_type = null) {
			if ($content == null) {
				/* Load content from file
				 */
				if (file_exists($filename) == false) {
					return false;
				}
				if (($content = file_get_contents($filename, FILE_BINARY)) == false) {	
					return false;
				}
				$content_type = mime_content_type($filename);
				$filename = basename($filename);
			}

			if ($content_type == null) {
				/* Determine content mimetype
				 */
				#$finfo = new finfo(FILEINFO_MIME);
				#$content_type = $finfo->buffer($content);
				$content_type = "application/octet-stream";
			}

			/* Add attachment
			 */
			array_push($this->attachments, array(
				"filename" => $filename,
				"content" => $content,
				"content_type" => $content_type));

			return true;
		}

		/* Send e-mail
		 *
		 * INPUT:  [string e-mail address recipient][, string name recipient]
		 * OUTPUT: boolean sending successful
		 * ERROR:  -
		 */
		public function send($to_address = null, $to_name = null) {
			if ($to_address !== null) {
				if ($this->valid_email($to_address) == false) {
					return false;
				}

				$this->to($to_address, $to_name);
			}

			if ((count($this->to) == 0) || ($this->message === null)) {
				return false;
			}

			if (count($this->attachments) > 0) {
				/* E-mail with attachments
				 */
				$headers = array("Content-Type: multipart/mixed; boundary=".$this->boundary);
				$message = "";

				/* Add attachments
				 */
				foreach ($this->attachments as $attachment) {
					$content = base64_encode($attachment["content"]);
					$content = wordwrap($content, 75, "\n", true);
					$message .=
						"--".$this->boundary."\n".
						"Content-Disposition: inline\n".
						"\tfilename=".$attachment["filename"]."\n".
						"Content-Type: ".$attachment["content_type"]."\n".
						"Content-Transfer-Encoding: base64\n\n".
						$content."\n\n";
				}

				/* Add message
				 */
				$message .= 
					"--".$this->boundary."\n".
					"Content-Type: ".$this->content_type."\n\n".
					$this->message."\n".
					"--".$this->boundary."--\n";
			} else {
				/* E-mail without attachments
				 */
				$headers = array("Content-Type: ".$this->content_type);
				$message = $this->message;
			}

			array_push($headers, "User-Agent: Banshee PHP framework e-mail library");

			/* Sender
			 */
			if ($this->from != null) {
				array_push($headers, "From: ".$this->from);
			}
			if ($this->reply_to != null) {
				array_push($headers, "Reply-To: ".$this->reply_to);
			}

			/* Carbon Copies
			 */
			if (count($this->cc) > 0) {	
				array_push($headers, "CC: ".implode(", ", $this->cc));
			}

			/* Blind Carbon Copies
			 */
			if (count($this->bcc) > 0) {	
				array_push($headers, "BCC: ".implode(", ", $this->bcc));
			}

			/* Secure mail headers
			 */
			foreach ($headers as &$header) {
				$header = str_replace("\n", "", $header);
				$header = str_replace("\r", "", $header);
			}

			/* Send the e-mail
			 */
			if (mail(implode(", ", $this->to), $this->subject, $message, implode("\n", $headers), "-f".$this->sender_address) == false) {
				return false;
			}

			$this->to = array();
			$this->cc = array();
			$this->bcc = array();

			return true;
		}
	}
?>
