<?php
	/* libraries/output.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	class output extends XML {
		private $page = null;
		private $language = DEFAULT_LANGUAGE;
		private $description = HEAD_DESCRIPTION;
		private $keywords = HEAD_KEYWORDS;
		private $messages = array();
		private $alternates = array();
		private $title = null;
		private $css_links = array();
		private $inline_css = null;
		private $content_type = "text/html; charset=utf-8";
		private $layout = "layout_site";

		/* Constructor
		 *
		 * INPUT:  object database, boolean AJAX request
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($db, $page) {
			ob_start();
			set_exception_handler(array($this, "exception_handler"));
			set_error_handler(array($this, "error_handler"), E_ALL & ~E_NOTICE);

			$this->page = $page;
			parent::__construct($db);

			/* Page layout
			 */
			if (substr($this->page->url, 0, 6) == "/admin") {
				$this->layout = "layout_cms";
			} else if ($this->page->module == "logout") {
				if ($_SESSION["previous_layout"] == "layout_cms") {
					$this->layout = "layout_cms";
				}
			}
		}

		/* Constructor
		 *
		 * INPUT:  object database, boolean AJAX request
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __destruct() {
			$_SESSION["previous_layout"] = $this->layout;
		}

		/* Magic method get
		 *
		 * INPUT:  string key
		 * OUTPUT: mixed value
		 * ERROR:  null
		 */
		public function __get($key) {
			switch ($key) {
				case "language": return $this->language;
				case "description": return $this->description;
				case "keywords": return $this->keywords;
				case "title": return $this->title;
				case "inline_css": return $this->inline_css;
				case "content_type": return $this->content_type;
				case "layout": return $this->layout;
			}

			return parent::__get($key);
		}

		/* Magic method set
		 *
		 * INPUT:  string key, string value
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __set($key, $value) {
			switch ($key) {
				case "language": $this->language = $value; break;
				case "description": $this->description = $value; break;
				case "keywords": $this->keywords = $value; break;
				case "title": $this->title = $value; break;
				case "inline_css": $this->inline_css = $value; break;
				case "content_type": $this->content_type = $value; break;
				default: trigger_error("Unknown output variable: ".$key);
			}
		}

		/* Exception handler
		 *
		 * INPUT:  error object
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function exception_handler($e) {
			print "Caught exception '".$e->getmessage()."'<br />\n";
		}

		/* Error handler
		 *
		 * INPUT:  int error number, string error string, string filename, int line number
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function error_handler($errno, $errstr, $errfile, $errline) {
			print $errstr." in ".$errfile." on line ".$errline.".<br />\n";
		}

		/* Add CSS link to output
		 *
		 * INPUT:  string CSS filename
		 * OUTPUT: boolean CSS file exists
		 * ERROR:  -
		 */
		public function add_css($css) {
			if (file_exists("css/".$css) == false) {
				return false;
			}

			array_push($this->css_links, $css);

			return true;
		}

		/* Add alternate link
		 *
		 * INPUT:  string title, string type, string url
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function add_alternate($title, $type, $url) {
			array_push($this->alternates, array(
				"title" => $title,
				"type"  => $type,
				"url"   => $url));
		}

		/* Set page layout
		 *
		 * INPUT:  string layout
		 * OUTPUT: bool layout accepted
		 * ERROR:  -
		 */
		public function set_layout($layout) {
			if (file_exists("../views/includes/".$layout.".xslt") == false) {
				return false;
			}

			$this->layout = $layout;

			return true;
		}

		/* Add message to message buffer
		 *
		 * INPUT:  string message, int message type
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function add_message($message, $type = 0) {
			array_push($this->messages, array(
				"content" => $message,
				"type"    => $type));
		}

		/* Close XML tag
		 *
		 * INPUT:  -
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function close_tag() {
			if (($this->page->ajax_request == false) && ($this->depth == 1)) {
				/* Messages
				 */
				$this->open_tag("messages");
				foreach ($this->messages as $message) {
					$this->add_tag("message", $message["content"], array("type" => $message["type"]));
				}
				$this->close_tag();

				$this->open_tag($this->layout);

				/* Header information
				 */
				$this->add_tag("description", $this->description);
				$this->add_tag("keywords", $this->keywords);
				$this->add_tag("title", ($this->title != null ? $this->title." - " : "").HEAD_TITLE);
				$this->add_tag("language", $this->language);

				/* Cascading Style Sheets
				 */
				$this->open_tag("styles");
				foreach ($this->css_links as $css) {
					$this->add_tag("style", $css);
				}
				$this->close_tag();
				$this->add_tag("inline_css", $this->inline_css);

				/* Alternates
				 */
				$this->open_tag("alternates");
				foreach ($this->alternates as $alternate) {
					$this->add_tag("alternate", $alternate["title"], array(
						"type"  => $alternate["type"],
						"url" => $alternate["url"]));
				}
				$this->close_tag();

				/* Google Analytics
				 */
				if (GOOGLE_ANALYTICS != "") {
					$this->add_tag("google_analytics", GOOGLE_ANALYTICS);
				}

				$this->close_tag();
			}

			parent::close_tag();
		}

		/* Generate output via XSLT
		 *
		 * INPUT:  string output type, string XSLT file
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function generate($output, $view = null) {
			$errors = ob_get_contents();
			ob_end_clean();

			switch ($output) {
				case "xml":
					header("Content-Type: text/xml");
					print $this->document;
					break;
				case "data":
					header("Content-Type: text/plain");
					if (is_true(DEBUG) && (strlen($errors) > 0)) {
						print $errors."\n".str_repeat("-", 80)."\n";
					}
					print $this->document;
					break;
				case null:
					$this->set_xslt_parameter("layout", "site");

					if (($html = $this->transform($view)) === false) {
						print "XSL Transformation error";
						return;
					}

					if (strlen($errors) > 0) {
						$errors = str_replace("<br />", "", $errors);
						if (is_true(DEBUG)) { 
							$html .= "<fieldset id=\"internal_errors\" class=\"internal_errors\">";
							$html .= "<legend>Internal errors <img src=\"/images/cross.gif\" title=\"Close message box\" alt=\"Close\" class=\"close\" onClick=\"javascript:document.getElementById('internal_errors').style.display = 'none'\"></legend>\n";

							$errors = htmlentities($errors);
							$errors = str_replace("\t", "    ", $errors);
							$errors = explode("\n", $errors);
							foreach ($errors as $error) {
								$len = strlen($error);
								$error = ltrim($error);
								if (($len = $len - strlen($error)) > 0) {
									$html .= str_repeat("&nbsp;", $len);
								}
								$html .= $error."<br />\n";
							}

							$html .= "</fieldset>\n";
						} else {
							$message =
								"Date, time: ".date("j F Y, H:i:s")."\n".
								"Used URL  : ".$_SERVER["REQUEST_URI"]."\n".
								"IP address: ".$_SERVER["REMOTE_ADDR"]."\n".
								"User-Agent: ".$_SERVER["HTTP_USER_AGENT"]."\n".
								"\n".$errors;

							$email = new email("Internal error at ".$_SERVER["SERVER_NAME"], "no-reply@".$_SERVER["SERVER_NAME"]);
							$email->message($message);
							$email->send(WEBMASTER_EMAIL);
						}
					}

					/* GZip content encoding
					 */
					$encodings = $_SERVER["HTTP_ACCEPT_ENCODING"];
					$php_gzip = ini_get("zlib.output_compression");
					if (($encodings !== null) && (strlen($html) >= 1024) && is_false($php_gzip)) {
						$encodings = explode(",", $encodings);
						foreach ($encodings as $encoding) {
							$encoding = trim($encoding);
							if ($encoding == "gzip") {
								header("Content-Encoding: gzip");
								$html = gzencode($html, 6);
							}
						}
					}

					/* Print output
					 */
					header("Content-Type: ".$this->content_type);
					header("Content-Language: ".$this->language);
					if (is_false($php_gzip)) {
						header("Content-Length: ".strlen($html));
					}
					print $html;
					break;
				default:
					print "Unknown output type";
			}
		}
	}
?>
