<?php
	/* libraries/page.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 *
	 * Don't change this file, unless you know what you are doing.
	 */

	class page {
		private $db = null;
		private $user = null;
		private $module = null;
		private $url = null;
		private $page = null;
		private $type = "";
		private $http_code = 200;
		private $is_public = true;
		private $pathinfo = array();
		private $ajax_request = null;

		/* Constructor
		 *
		 * INPUT:  object database, object user, boolean AJAX request
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($db, $user, $ajax_request = false) {
			$this->db = $db;
			$this->user = $user;
			$this->ajax_request = $ajax_request;

			/* Prevent Cross-Site Request Forgery
			 */
			if ($_SERVER["REQUEST_METHOD"] == "POST") {
				list($protocol,, $referer_host) = explode("/", $_SERVER["HTTP_REFERER"]);
				if (($protocol == "http:") && ($_SERVER["HTTP_HOST"] != $referer_host)) {
					$user->logout();
					$_SERVER["REQUEST_METHOD"] = "GET";
					$_GET = array();
					$_POST = array();
				}
			}

			if (is_false(WEBSITE_ONLINE) && ($_SERVER["REMOTE_ADDR"] != WEBSITE_ONLINE)) {
				$page = "offline";
			} else if ($db->connected == false) {
				$page = "error";
				$this->http_code = 500;
			} else {
				list($this->url) = explode("?", $_SERVER["REQUEST_URI"], 2);
				if (($this->url == "/index.php") || ($this->url == "/ajax.php")) {
					$page = isset($_GET["page"]) ? $_GET["page"] : START_PAGE;
					$this->pathinfo[0] = $page;
				} else {
					$path = trim($this->url, "/");
					$page = valid_input($path, VALIDATE_URL, VALIDATE_NONEMPTY) ? $path : START_PAGE;
					$this->pathinfo = secure_input(explode("/", $page));
				}
			}

			$this->select_module($page);
		}

		/* Desctructor
		 *
		 * INPUT:  -
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __destruct() {
			$_SESSION["last_visit"] = time();
		}
		
		/* Magic method get
		 *
		 * INPUT:  string key 
		 * OUTPUT: mixed value
		 * ERROR:  null
		 */
		public function __get($var) {
			switch ($var) {
				case "module": return $this->module;
				case "url": return $this->url;
				case "page": return $this->page !== null ? $this->page : $this->module;
				case "type": return ltrim($this->type, ".");
				case "view": return $this->module.$this->type;
				case "pathinfo": return $this->pathinfo;
				case "http_code": return $this->http_code;
				case "ajax_request": return $this->ajax_request;
				case "is_public": return $this->is_public;
				case "is_private": return $this->is_public == false;
				case "logged_in": return $this->logged_in;
			}

			return null;
		}

		/* Page available on disk
		 *
		 * INPUT:  string URL, string page configuration file
		 * OUTPUT: string module identifier
		 * ERROR:  -
		 */
		private function page_on_disk($url, $page_file) {
			$module = null;
			$url = explode("/", $url);
			$url_count = count($url);

			foreach (config_file($page_file) as $line) {
				$page = explode("/", $line);
				$parts = count($page);
				$match = true;

				for ($i = 0; $i < $parts; $i++) {
					if ($page[$i] == "*") {
						continue;
					} else if ($page[$i] !== $url[$i]) {
						$match = false;
						break;
					}
				}

				if ($match && (strlen($line) >= strlen($module))) {
					$module = page_to_module($line);
				}
			}

			/* Determine page type
			 */
			if ($module !== null) {
				if (($pos = strrpos($module, ".")) !== false) {
					$this->type = substr($module, $pos);
					$module = substr($module, 0, $pos);
				}
			}

			return $module;
		}

		/* Page available in database
		 *
		 * INPUT:  string URL, int page type
		 * OUTPUT: string module identifier
		 * ERROR:  -
		 */
		private function page_in_database($url, $private) {
			$query = "select id,visible from pages where url=%s and private=%d limit 1";
			if (($result = $this->db->execute($query, "/".$url, $private)) == false) {
				return null;
			}

			if ($result[0]["visible"] == NO) {
				if ($this->user->access_allowed("admin/page") == false) {
					return null;
				}
			}
			$this->url = "/".$url;
			$this->page = $url;

			return PAGE_MODULE;
		}

		/* Determine what module needs te be loaded based on requested page
		 *
		 * INPUT:  string page identifier
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function select_module($page) {
			if (($this->module !== null) && ($this->module !== "login")) {
				return;
			}

			/* Page aliases
			 */
			foreach (config_file("page_aliases") as $line) {
				list($from, $to) = explode("->", $line);
				
				$from = trim($from);
				$length = strlen($from);

				if (substr($page."/", 0, $length + 1) === substr($from."/", 0, $length + 1)) {
					$page = trim($to).substr($page, $length);
					$this->url = "/".$page;
					$this->page = $from;
					break;
				}
			}

			/* AJAX page
			 */
			if ($this->ajax_request) {
				if (($this->module = $this->page_on_disk($page, "ajax_pages")) !== null) {
					return;
				}
			}

			/* Public page
			 */
			if (($this->module = $this->page_on_disk($page, "public_pages")) !== null) {
				return;
			} else if (($this->module = $this->page_in_database($page, NO)) !== null) {
				return;
			}

			/* Change profile before access to private pages
			 */
			if ($this->user->logged_in && ($page != "logout")) {
				if ($this->user->status == USER_STATUS_CHANGEPWD) {
					$page = "profile";
					$this->type = "";
				}
			}

			/* Private page
			 */
			if (($this->module = $this->page_on_disk($page, "private_pages")) === null) {
				$this->module = $this->page_in_database($page, YES);
			}

			if ($this->module == null) {
				/* Page does not exist.
				 */
				$this->module = "error";
				$this->http_code = 404;
				$this->type = "";
			} else if ($this->user->logged_in == false) {
				/* User not logged in.
				 */
				$this->module = "login";
				$this->type = "";
			} else if ($this->user->access_allowed($this->__get("page").$this->type) == false) {
				/* Access denied because not with right role.
				 */
				$this->module = "error";
				$this->http_code = 401;
				$this->type = "";
			} else {
				/* Access allowed.
				 */
				$this->is_public = false;
				$_SESSION["last_private_visit"] = time();
			}
		}
	}
?>
