<?php
	/* libraries/settings.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	/* For internal usage. Only change if you know what you're doing!
	 */
	define("BANSHEE_VERSION", "1.3");
	define("MONITOR_VERSION", "0.2");
	define("ADMIN_ROLE_ID", 1);
	define("YES", 1);
	define("NO", 0);
	define("USER_STATUS_DISABLED", 0);
	define("USER_STATUS_CHANGEPWD", 1);
	define("USER_STATUS_ACTIVE", 2);
	define("SESSION_NAME", "WebsiteSessionID");
	define("PAGE_MODULE", "includes/page");
	define("DAY", 86400);

	/* Hiawatha Monitor settings
	 */
	define("MONITOR_DAYS", 30);
	define("ALERT_HOURS", 6);

	/* Pre-defined validation strings for valid_input()
	 */
	define("VALIDATE_CAPITALS",		"ABCDEFGHIJKLMNOPQRSTUVWXYZ");
	define("VALIDATE_NONCAPITALS",	"abcdefghijklmnopqrstuvwxyz");
	define("VALIDATE_LETTERS",		VALIDATE_CAPITALS.VALIDATE_NONCAPITALS);
	define("VALIDATE_PHRASE",		VALIDATE_LETTERS." ,.?!:;-'");
	define("VALIDATE_NUMBERS",		"0123456789");
	define("VALIDATE_EMAIL",		VALIDATE_LETTERS.VALIDATE_NUMBERS."_-@.");
	define("VALIDATE_SYMBOLS",		"!@#$%^&*()_-+={}[]|\:;\"'`~<>,./?");
	define("VALIDATE_URL",          VALIDATE_LETTERS.VALIDATE_NUMBERS."-_/.");

	define("VALIDATE_NONEMPTY",     0);

	$allowed_uploads = array("jpg", "jpeg", "gif", "png", "pdf", "doc", "xls", "zip", "txt");
	$months_of_year = array("january", "february", "march", "april", "may", "june",
		"july", "august", "september", "october", "november", "december");
	$days_of_week = array("monday", "tuesday", "wednesday", "thursday", "friday",
		"saturday", "sunday");

	/* Class for settings in database
	 */
	class settings {
		private $db = null;
		private $max_value_len = 256;
		private $cache = array();

		/* Constructor
		 *
		 * INPUT:  object database
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($db) {
			$this->db = $db;
		}

		/* Set setting
		 *
		 * INPUT:  string key, mixed value
		 * OUTPUT: boolean set successful
		 * ERROR:  -
		 */
		public function __set($id, $value) {
			if ($value === null) {	
				$result = $this->db->delete("settings", $id);

				if ($result !== false) {
					unset($this->cache[$id]);
				}
			} else {
				$ser_value = serialize($value);
				if (strlen($ser_value) > $this->max_value_len) {
					return false;
				}

				if ($this->$id === null) {
					$result = $this->db->insert("settings", array("id" => $id, "value" => $ser_value));
				} else {
					$result = $this->db->update("settings", $id, array("value" => $ser_value));
				}

				if ($result !== false) {
					$this->cache[$id] = $value;
				}

			}

			return $result !== false;
		}

		/* Get setting
		 *
		 * INPUT:  string key
		 * OUTPUT: mixed value
		 * ERROR:  null
		 */
		public function __get($id) {
			if (isset($this->cache[$id])) {
				return $this->cache[$id];
			} else if (($setting = $this->db->entry("settings", $id)) == false) {
				return null;
			}

			$value = unserialize($setting["value"]);
			$this->cache[$id] = $value;

			return $value;
		}
	}

	/* Auto class loader
	 *
	 * INPUT:  string class name
	 * OUTPUT: -
	 * ERROR:  -
	 */
	function __autoload($class_name) {
		$rename = array(
			"HTTPS"       => "HTTP",
			"HTTP_proxy"  => "HTTP",
			"HTTPS_proxy" => "HTTP",
			"POP3S"       => "POP3");
		if (isset($rename[$class_name])) {
			$class_name = $rename[$class_name];
		}

		$locations = array("libraries", "libraries/database");
		foreach ($locations as $location) {
			if (file_exists($file = "../".$location."/".strtolower($class_name).".php")) {
				include_once($file);
				break;
			}
		}
	}

	function check_PHP_setting($key, $value) {
		if (ini_get($key) != $value) {
			exit("Please, set the PHP flag '".$key."' to '".$value."'!\n");
		}
	}

	/* Load configuration file
	 *
	 * INPUT:  string configurationfile
	 * OUTPUT: array( key => value[, ...] )
	 * ERROR:  -
	 */
	function config_file($file) {
		static $cache = array();

		if (isset($cache[$file])) {
			return $cache[$file];
		}

		$config_file = "../settings/".$file.".conf";
		if (file_exists($config_file) == false) {
			return array();
		}

		$config = array();
		foreach (file($config_file) as $line) {
			if (($line = preg_replace("/#.*/", "", trim($line))) !== "") {
				array_push($config, $line);
			}
		}

		$cache[$file] = $config;

		return $config;
	}

	/* Convert mixed to boolean
	 *
	 * INPUT:  mixed
	 * OUTPUT: boolean
	 * ERROR:  -
	 */
	function is_true($bool) {
		return in_array($bool, array(true, YES, "1", "yes", "true", "on"), true);
	}

	/* Convert mixed to boolean
	 *
	 * INPUT:  mixed
	 * OUTPUT: boolean
	 * ERROR:  -
	 */
	function is_false($bool) {
		return (is_true($bool) === false);
	}

	/* Convert boolean to string
	 *
	 * INPUT:  boolean
	 * OUTPUT: string "yes"|"no"
	 * ERROR:  -
	 */
	function show_boolean($bool) {
		return (is_true($bool) ? "yes" : "no");
	}

	/* Parse website.conf
	 */
	foreach (config_file("website") as $line) {
		list($key, $value) = explode("=", chop($line), 2);
		define(trim($key), trim($value));
	}

	/* PHP settings
	 */
	set_magic_quotes_runtime(0);
?>
