<?php
	/* libraries/xml.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.hiawatha-webserver.org/banshee
	 */

	define("XML_HEADER", "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>");

	class XML {
		protected $db = null;
		protected $xml_data = "";
		private $xslt_parameters = array();
		private $open_tags = array();
		private $cache_id = null;
		private $cache_buffer = "";
		private $cache_timeout;

		/* Constructor
		 *
		 * INPUT:  [object database]
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function __construct($db = null) {
			$this->db = $db;
		}

		/* Magic method get
		 *
		 * INPUT:  string key
		 * OUTPUT: mixed value
		 * ERROR:  null
		 */
		public function __get($key) {
			switch ($key) {
				case "depth": return count($this->open_tags);
				case "data": return $this->xml_data;
				case "document": return XML_HEADER."\n".$this->xml_data;
			}

			return null;
		}

		/* Translate special characters in string to XML entities
		 *
		 * INPUT:  string data
		 * OUTPUT: string data
		 * ERROR:  -
		 */
		private function xmlspecialchars($str) {
			$str  = utf8_encode($str);
			$from = array("&", "\"", "'", "<", ">");
			$to   = array("&amp;", "&quot;", "&apos;", "&lt;", "&gt;");

			return str_replace($from, $to, $str);
		}

		/* Add string to buffer
		 *
		 * INPUT:  string data
		 * OUTPUT: -
		 * ERROR:  -
		 */
		private function add_to_buffer($str) {
			$this->xml_data .= $str;

			if ($this->cache_id !== null) {
				$this->cache_buffer .= $str;
			}
		}

		/* Add open-tag to buffer
		 *
		 * INPUT:  string tag, array( string data[, ...] )[, boolean append EOL]
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function open_tag($tag, $params = array(), $eol = true) {
			$this->add_to_buffer("<".$tag);
			foreach ($params as $key => $value) {
				$this->add_to_buffer(" ".$key."=\"".$this->xmlspecialchars($value)."\"");
			}
			$this->add_to_buffer(">".($eol ? "\n" : ""));

			array_push($this->open_tags, $tag);
		}

		/* Add close-tag to buffer
		 *
		 * INPUT:  -
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function close_tag() {
			if (count($this->open_tags) > 0) {
				$this->add_to_buffer("</".array_pop($this->open_tags).">\n");
			}
		}

		/* Add tag to buffer
		 *
		 * INPUT:  string tag, string data, array( string key => string value[, ...] )
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function add_tag($tag, $data = null, $params = array()) {
			$this->open_tag($tag, $params, false);
			if ($data !== null) {
				$this->add_to_buffer($this->xmlspecialchars($data));
			}
			$this->close_tag();
		}

		/* Add record to buffer
		 *
		 * INPUT:  array( string key => string value[, ...] ), string tag name, array( string key => string value[, ...] )
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function record($record, $name = null, $params = array()) {
			if ($name !== null) {
				if (isset($record["id"])) {
					$params["id"] = $record["id"];
				}
				$this->open_tag($name, $params);
			}

			$skip_tags = array("id", "password");
			foreach (array_keys($record) as $key) {
				if ((in_array($key, $skip_tags) == false) && (is_array($record[$key]) == false)) {
					$this->add_tag($key, $record[$key]);
				}
			}
			if ($name !== null) {
				$this->close_tag($name);
			}
		}

		/* Add XML data to buffer
		 *
		 * INPUT:  XML data, string tat name
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function add_xml($xml, $tag) {
			if (($str = $xml->saveXML()) === false) {
				return;
			}

			if (($begin = strpos($str, "<".$tag.">")) === false) {
				return;
			}
			if (($end = strrpos($str, "</".$tag.">")) === false) {
				return;
			}
			$end += strlen($tag) + 3;

			$str = substr($str, $begin, $end - $begin)."\n";

			$this->add_to_buffer($str);
		}

		/* Set XSLT parameter
		 *
		 * INPUT:  string key, string value
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function set_xslt_parameter($key, $value) {
			$this->xslt_parameters[$key] = $value;
		}

		/* Perform XSL transformation
		 *
		 * INPUT:  string XSLT filename
		 * OUTPUT: string XSLT result
		 * ERROR:  false
		 */
		public function transform($xslt_file) {
			$xslt = new DomDocument();
			$xml = new DomDocument();

			if ($xslt->load("../views/".$xslt_file.".xslt") == false) {
				return false;
			}
			
			if ($xml->loadXML($this->document) == false) {
				return false;
			}

			$processor = new XsltProcessor();
			$processor->importStylesheet($xslt);

			foreach ($this->xslt_parameters as $key => $value) {
				$processor->setParameter("", $key, $value);
			}

			return $processor->transformToXML($xml);
		}

		/* Start caching XML additions done after this call
		 *
		 * INPUT:  string cache identifier[, int cache timeout in seconds]
		 * OUTPUT: boolean start caching successful
		 * ERROR:  -
		 */
		public function start_caching($id, $timeout = CACHE_TIMEOUT) {
			if (($this->db === null) || ($this->cache_id !== null)) {
				return false;
			}

			$this->cache_id = $id;
			$this->cache_timeout = date("Y-m-d H:i:s", strtotime("+".$timeout." seconds"));

			return true;
		}

		/* Stop caching
		 *
		 * INPUT:  -
		 * OUTPUT: boolean stop caching successfull
		 * ERROR:  -
		 */
		public function stop_caching() {
			if (($this->db === null) || ($this->cache_id === null)) {
				return false;
			}

			if ($this->remove_from_cache($this->cache_id) === false) {
				return false;
			}

			$data = array(
				"id"      => $this->cache_id,
				"content" => $this->cache_buffer,
				"timeout" => $this->cache_timeout);
			if (($this->db->insert("cache", $data)) == false) {
				return false;
			}

			$this->abort_caching();

			return true;
		}

		/* Abort XML caching
		 *
		 * INPUT:  -
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function abort_caching() {	
			$this->cache_id = null;
			$this->cache_buffer = "";
		}

		/* Fetch XML from cache and append to buffer
		 *
		 * INPUT:  string cache identifier
		 * OUTPUT: -
		 * ERROR:  -
		 */
		public function fetch_from_cache($id) {
			if ($this->db === null) {
				return false;
			}

			$query = "select * from cache where id=%s and timeout>now() limit 1";
			if (($result = $this->db->execute($query, $id)) == false) {	
				return false;
			}

			$this->xml_data .= $result[0]["content"];

			return true;
		}

		/* Remote entry from XML cache
		 *
		 * INPUT:  string cache identifier
		 * OUTPUT: boolean removal successful
		 * ERROR:  -
		 */
		public function remove_from_cache($id) {
			if ($this->db === null) {
				return false;
			}

			$query = "delete from cache where id=%s";
			return $this->db->query($query, $id) !== false;
		}

		/* Clear entire cache
		 *
		 * INPUT:  -
		 * OUTPUT: boolean clear successful
		 * ERROR:  -
		 */
		public function clear_cache() {
			if ($this->db === null) {
				return false;
			}

			$query = "truncate table %S";
			return $this->db->query($query, "cache") !== false;
		}
	}
?>
