<?php
	/* libraries/general.php
	 *
	 * Copyright (C) by Hugo Leisink <hugo@leisink.net>
	 * This file is part of the Banshee PHP framework
	 * http://www.banshee-php.org/
	 */

	/* Convert mixed to boolean
	 *
	 * INPUT:  mixed
	 * OUTPUT: boolean
	 * ERROR:  -
	 */
	function is_true($bool) {
		return in_array($bool, array(true, YES, "1", "yes", "true", "on"), true);
	}

	/* Convert mixed to boolean
	 *
	 * INPUT:  mixed
	 * OUTPUT: boolean
	 * ERROR:  -
	 */
	function is_false($bool) {
		return (is_true($bool) === false);
	}

	/* Convert boolean to string
	 *
	 * INPUT:  boolean
	 * OUTPUT: string "yes"|"no"
	 * ERROR:  -
	 */
	function show_boolean($bool) {
		return (is_true($bool) ? "yes" : "no");
	}

	/* Convert a page path to a module path
	 *
	 * INPUT:  array / string page path
	 * OUTPUT: array / string module path
	 * ERROR:  -
	 */
	function page_to_module($page) {
		if (is_array($page) == false) {
			$page = str_replace("*/", "", $page);

			if (($pos = strrpos($page, ".")) !== false) {
				$page = substr($page, 0, $pos);
			}
		} else foreach ($page as &$item) {
			$item = page_to_module($item);
		}

		return $page;
	}

	/* Convert a page path to a module path
	 *
	 * INPUT:  array / string page path
	 * OUTPUT: array / string page type
	 * ERROR:  -
	 */
	function page_to_type($page) {
		if (is_array($page) == false) {
			if (($pos = strrpos($page, ".")) !== false) {
				$page = substr($page, $pos);
			} else {
				$page = "";
			}
		} else foreach ($page as &$item) {
			$item = page_to_type($item);
		}

		return $page;
	}

	/* Get users in group
	 *
	 * INPUT:  string group name
	 * OUTPUT: array user
	 * ERROR:  false
	 */
	function users_in_group($db, $group) {
		$query = "select distinct u.* from users u, user_role m, roles r ".
		         "where r.name=%s and r.id=m.role_id and m.user_id=u.id";

		return $db->execute($query, $group);
	}

	/* Flatten array to new array with depth 1
	 *
	 * INPUT:  array
	 * OUTPUT: array
	 * ERROR:  -
	 */
	function array_flatten($data) {
		$result = array();
		foreach ($data as $item) {
			if (is_array($item)) {
				$result = array_merge($result, array_flatten($item));
			} else {
				array_push($result, $item);
			}
		}

		return $result;
	}

	/* Match if IP matches IP subnet
	 *
	 * INPUT:  string ip, string net
	 * OUTPUT: bool
	 * ERROR:  -
	 */
	function ip_match($ip, $net) {
		list($net, $mask) = explode("/", $net, 2);
		if ($mask == "") {
			$mask = 0;
		} else {
			$mask = 32 - (int)$mask;
		}

		$ip = explode(".", $ip);
		$ip = ((int)$ip[0] << 24) + ((int)$ip[1] << 16) + ((int)$ip[2] << 8) + (int)$ip[3];
		$ip = $ip >> $mask;

		$net = explode(".", $net);
		$net = ((int)$net[0] << 24) + ((int)$net[1] << 16) + ((int)$net[2] << 8) + (int)$net[3];
		$net = $net >> $mask;

		return $ip == $net;
	}

	/* Prepare string for unescaped output
	 *
	 * INPUT:  string data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	function unescaped_output($str) {
		$str = htmlentities($str);
		$str = str_replace("\r", "", $str);
		$str = str_replace("\n", "<br />", $str);

		return $str;
	}

	/* Prepare data for CSV output
	 *
	 * INPUT:  string/array data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	function csv_output($data) {
		if (is_array($data) == false) {
			$data = str_replace('"', '""', $data);
			$data = '"'.$data.'"';
		} else {
			foreach ($data as &$item) {
				$item = csv_output($item);
			}
			$data = implode(",", $data);
		}

		return $data;
	}

	/* Prepare string for iCalendar output
	 *
	 * INPUT:  string data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	function ics_output($str) {
		$str = str_replace("\\", "\\\\", $str);
		$str = str_replace("\"", "\\\"", $str);
		$str = str_replace("\r", "", $str);
		$str = str_replace("\n", "\\n", $str);

		return $str;
	}

	/* Generate link from text
	 *
	 * INPUT:  string text
	 * OUTPUT: string link
	 * ERROR:  -
	 */
	function make_link($text) {
		$result = "";
		$len = strlen($text);
		for ($i = 0; $i < $len; $i++) {
			if (($text[$i] >= "a") && ($text[$i] <= "z")) {
				$result .= $text[$i];
			} else if (($text[$i] >= "A") && ($text[$i] <= "Z")) {
				$result .= strtolower($text[$i]);
			} else if (($text[$i] >= "0") && ($text[$i] <= "9")) {
				$result .= $text[$i];
			} else if ($text[$i] == " ") {
				$result .= "-";
			}
		}

		return $result;
	}

	/* Truncate text
	 *
	 * INPUT:  string text, int length
	 * OUTPUT: string truncated text
	 * ERROR:  -
	 */
	function truncate_text($text, $length) {
		if (strlen($text) <= $length) {
			return $text;
		}

		$is_space = ($text[$length] === " ");
		$text = substr($text, 0, $length);
		if ($is_space == false) {
			if (($pos = strrpos($text, " ")) !== false) {
				$text = substr($text, 0, $pos);
			}
		}

		return $text."...";
	}

	/* Truncate HTML
	 *
	 * INPUT:  string HTML, int length
	 * OUTPUT: string truncated HTML
	 * ERROR:  -
	 */
	function truncate_html($html, $length) {
		$open_tags = array();
		$html_len = strlen($html);

		for ($i = 0; $i < $html_len; $i++) {
			if ($html[$i] == "<") {
				$name_begin = $i + 1;
				if (($tag_end = strpos($html, ">", $name_begin)) === false) {
					continue;
				}
				$i = $tag_end;

				if ($html[$tag_end - 1] == "/") {
					continue;
				}

				if ($open_tag = ($html[$name_begin] == "/")) {
					array_pop($open_tags);
				} else {
					if (($name_end = strpos($html, " ", $name_begin)) === false) {
						$name_end = $tag_end;
					} else if ($name_end > $tag_end) {
						$name_end = $tag_end;
					}

					$tag = substr($html, $name_begin, $name_end - $name_begin);
					array_push($open_tags, $tag);
				}
			} else if (--$length == 0) {
				break;
			}
		}

		$html = substr($html, 0, $i + 1);
		if ($length == 0) {
			$html .= "...";
		}

		$open_tags = array_reverse($open_tags);
		foreach ($open_tags as $tag) {
			$html .= "</".$tag.">";
		}

		return $html;
	}

	/* Perform HTTP GET request and follow redirects
	 *
	 * INPUT:  string URL[, int redirect limit]
	 * OUTPUT: array HTTP result
	 * ERROR:  false
	 */
	function follow_http_redirects($url, $redirects = 5) {
		$result = false;

		while ($redirects-- >= 0) {
			list($protocol,, $hostname, $path) = explode("/", $url, 4);

			if ($protocol == "http:") {
				$http = new http($hostname);
			} else if ($protocol == "https:") {
				$http = new https($hostname);
			} else {
				break;
			}

			$result = $http->GET("/".$path);
			unset($http);

			if ($result === false) {
				break;
			} else if (($result["status"] != 301) && ($result["status"] != 302)) {
				break;
			} else if (($url = $result["headers"]["Location"]) == "") {
				break;
			}
		}

		return $result;
	}

	/* Decode a GZip encoded string
	 *
	 * INPUT:  string GZip data
	 * OUTPUT: string data
	 * ERROR:  -
	 */
	if (function_exists("gzdecode") == false) {

	function gzdecode($data) {
		$file = tempnam("/tmp", "gzip");

		@file_put_contents($file, $data);
		ob_start();
		readgzfile($file);
		$data = ob_get_clean();
		unlink($file);

		return $data;
	}

	}
?>
